package com.install4j.api.update;

import com.install4j.api.beans.ErrorHandlingCallback;
import com.install4j.api.context.UserCanceledException;
import com.install4j.runtime.installer.helper.apiimpl.UpdateCheckerImpl;
import com.install4j.runtime.launcher.integration.AutomaticUpdate;
import com.install4j.runtime.launcher.integration.UpdateExecutionConfig;
import com.install4j.runtime.util.VersionCheck;

import java.io.IOException;
import java.util.List;
import java.util.Objects;

/**
 * Utility class to download an update descriptor programmatically from a given URL. In an installer
 * application, it is recommended to use the "Check for update" action instead. This class is mainly
 * intended for checking for updates in your application.
 * <p>In order to use this class in your application, please include {@code resource/i4jruntime.jar} from your
 * install4j installation into your class path. You do not need to distribute this file
 * along with your application, install4j will do this automatically for you.
 * </p>
 *
 * @author ej-technologies GmbH
 * @see UpdateDescriptor
 */
public class UpdateChecker {

    /**
     * Convenience method for {@link #getUpdateDescriptor(UpdateCheckRequest)}.
     *
     * @param urlSpec               see {@link UpdateCheckRequest#urlSpec(String)}
     * @param displayMode           see {@link UpdateCheckRequest#applicationDisplayMode(ApplicationDisplayMode)}
     * @throws UserCanceledException if the user cancels the proxy dialog
     * @throws IOException           if the download fails
     */
    public static UpdateDescriptor getUpdateDescriptor(String urlSpec, ApplicationDisplayMode displayMode) throws UserCanceledException, IOException {
        return getUpdateDescriptor(urlSpec, displayMode, null);
    }

    /**
     * Convenience method for {@link #getUpdateDescriptor(UpdateCheckRequest)}.
     *
     * @param urlSpec               see {@link UpdateCheckRequest#urlSpec(String)}
     * @param displayMode           see {@link UpdateCheckRequest#applicationDisplayMode(ApplicationDisplayMode)}
     * @param errorHandlingCallback see {@link UpdateCheckRequest#errorHandlingCallback(ErrorHandlingCallback)}
     * @throws UserCanceledException if the user cancels the proxy dialog
     * @throws IOException           if the download fails
     */
    public static UpdateDescriptor getUpdateDescriptor(String urlSpec, ApplicationDisplayMode displayMode, ErrorHandlingCallback errorHandlingCallback) throws UserCanceledException, IOException {
        UpdateCheckRequest updateCheckRequest = new UpdateCheckRequest(urlSpec).applicationDisplayMode(displayMode).errorHandlingCallback(errorHandlingCallback);
        return getUpdateDescriptor(updateCheckRequest);
    }


    /**
     * Download an {@code updates.xml} file programatically from a given URL and returns an
     * {@link UpdateDescriptor} instance. The {@link UpdateCheckRequest} specifies the URL and other connection options.
     *
     * @param updateCheckRequest the update check request
     * @return the update descriptor
     * @throws UserCanceledException if the user cancels the proxy dialog
     * @throws IOException           if the download fails
     */
    public static UpdateDescriptor getUpdateDescriptor(UpdateCheckRequest updateCheckRequest) throws UserCanceledException, IOException {
        return UpdateCheckerImpl.getUpdateDescriptor(updateCheckRequest);
    }

    /**
     * Checks if the "Schedule update installation" action has registered a downloaded update for installation. This method returns {@code false}
     * if the update won't be currently executed because of a failed previous installation or because the installer is currently running. If an
     * update fails or is canceled, it will be rescheduled after 24 hours. The number of times the rescheduling happens can be configured with the
     * "Schedule update installation" action.
     *
     * @return {@code true} if scheduled and ready to be executed
     */
    public static boolean isUpdateScheduled() {
        return AutomaticUpdate.isScheduled(null, true, true);
    }

    /**
     * If an update is scheduled for installation, execute the update installer and shutdown the current launcher.
     * <p>
     * Equivalent to calling {@link #executeScheduledUpdate(List, boolean, List, Runnable)} with {@code null} for the {@code launcherArguments} parameter
     * </p>
     * @param updaterArguments updaterArguments to be passed to the updater.
     * @param restartLauncher if {@code true} a GUI or a console launcher will be restarted after the installation. If called from a service launcher, the
     *                        installer must take care of starting the service again. It can use {@code context.getBooleanVariable("sys.automaticUpdate")} if restart should be done conditionally.
     * @param shutdownRunnable if you want to invoke a custom routine for shutting down the current JVM, you can pass in a {@code Runnable}. Otherwise, {@code System.exit(0)} will be called.
     *                         If you pass in a Runnable, you must call {@code System.exit} at the end of your shutdown procedure.
     */
    public static void executeScheduledUpdate(List<String> updaterArguments, boolean restartLauncher, Runnable shutdownRunnable) {
        executeScheduledUpdate(updaterArguments, restartLauncher, null, shutdownRunnable);
    }

    /**
     * If an update is scheduled for installation, execute the update installer and shutdown the current launcher.
     *
     * @param updaterArguments updaterArguments to be passed to the updater.
     * @param restartLauncher if {@code true} a GUI or a console launcher will be restarted after the installation. If called from a service launcher, the
     *                        installer must take care of starting the service again. It can use {@code context.getBooleanVariable("sys.automaticUpdate")} if restart should be done conditionally.
     * @param launcherArguments if {@code restartLauncher} is {@code true} and this method is called within a GUI or a console launcher, the given arguments will be passed to the restarted launcher
     * @param shutdownRunnable if you want to invoke a custom routine for shutting down the current JVM, you can pass in a {@code Runnable}. Otherwise, {@code System.exit(0)} will be called.
     *                         If you pass in a Runnable, you must call {@code System.exit} at the end of your shutdown procedure.
     */
    public static void executeScheduledUpdate(List<String> updaterArguments, boolean restartLauncher, List<String> launcherArguments, Runnable shutdownRunnable) {
        AutomaticUpdate.checkUpdates(launcherArguments == null || launcherArguments.isEmpty() ? null : launcherArguments.toArray(new String[0]), new UpdateExecutionConfig(updaterArguments), restartLauncher, shutdownRunnable);
    }

    /**
     * Compares two version strings and checks if the first version is lower or equal than the second version.
     * This implementation is used by {@link UpdateDescriptor#getPossibleUpdateEntry()} and
     * {@link UpdateDescriptorEntry#checkVersionCompatible(String)} to check the compliance with the range of
     * the minimum and maximum updatable version.
     * @param expectedLowerVersion the version that is expected to be lower or equal
     * @param expectedHigherVersion the version that is expected to be higher or equal
     * @return {@code true} if {@code expectedHigherVersion} is higher or equal than {@code expectedLowerVersion}
     */
    public static boolean isVersionLessThanOrEqual(String expectedLowerVersion, String expectedHigherVersion) {
        return VersionCheck.checkCompatible(expectedLowerVersion, expectedHigherVersion);
    }

    /**
     * Compares two version strings and checks if the first version is greater than the second version.
     * This implementation is used by {@link UpdateDescriptor#getPossibleUpdateEntry()} and
     * {@link UpdateDescriptorEntry#checkVersionCompatible(String)} to check if a new version is greater
     * than the installed version.
     * @param expectedHigherVersion the version that is expected to be higher
     * @param expectedLowerVersion the version that is expected to be lower
     * @return {@code true} if {@code expectedHigherVersion} is higher than {@code installedVersion}
     */
    public static boolean isVersionGreaterThan(String expectedHigherVersion, String expectedLowerVersion) {
        return !Objects.equals(expectedLowerVersion, expectedHigherVersion) && isVersionLessThanOrEqual(expectedLowerVersion, expectedHigherVersion);
    }

}
