package com.install4j.api.styles;

import com.install4j.api.formcomponents.FormComponent;
import com.install4j.api.screens.FormPanelContainer;

import javax.swing.*;

/**
 * Helper for working with nested styles.
 * See {@link NestedStyleContainer} for more information.
 *
 * @author ej-technologies GmbH
 */
public interface StyleManager {
    /**
     * Get the screen style for a specified ID.
     * If the screen style is a {@link FormPanelContainer}, its form panel is not initialized.
     * Instead of calling {@link Style#createComponent()}, you have to call {@link #createStyleComponent(Style)} in order
     * to handle the creation of a form panel.
     *
     * @param styleId the ID of the screen style
     * @return the screen style
     */
    Style getStyleById(String styleId);

    /**
     * Clones a screen style for a specified ID. This is useful if you want to include multiple
     * instances of the same screen style within a single style.
     *
     * @param styleId the ID of the screen style
     * @return the cloned screen style
     */
    Style cloneStyleById(String styleId);

    /**
     * Create a screen style component.
     * If the screen style is a {@link FormPanelContainer}, a new form panel will be set on the screen style
     * before {@link Style#createComponent()} is called.
     *
     * @param style the screen style
     * @return the component
     */
    JComponent createStyleComponent(Style style);

    /**
     * Notify a nested screen style that it will be activated. You should call this method in the implementation of
     * {@link Style#willActivate()}. This will call {@link FormComponent#formWillActivate()} on all form components
     * if the nested screen style is a {@link FormPanelContainer}.
     *
     * @param style the nested screen style
     */
    void willActivate(Style style);

    /**
     * Notify a nested screen style that it has been activated. You should call this method in the implementation of
     * {@link Style#activated()}. This will call {@link FormComponent#formActivated()} on all form components
     * if the nested screen style is a {@link FormPanelContainer}.
     *
     * @param style the nested screen style
     */
    void activated(Style style);

    /**
     * Notify a nested screen style that it has been deactivated. You should call this method in the implementation of
     * {@link Style#deactivated()}. This will call {@link FormComponent#formDeactivated()} on all form components
     * if the nested screen style is a {@link FormPanelContainer}.
     *
     * @param style the nested screen style
     */
    void deactivated(Style style);

}
