package com.install4j.api.macos;

import com.install4j.api.ProcessInfo;
import com.install4j.api.Util;
import com.install4j.runtime.installer.InstallerConstants;
import com.install4j.runtime.installer.helper.InstallerUtil;
import com.install4j.runtime.installer.helper.RunningProcessChecker;
import com.install4j.runtime.installer.platform.macos.MacProcessHelper;

import java.util.Collection;

/**
 * Collection of static methods to check for running processes on macOS and terminate them.
 * @author ej-technologies GmbH
 */
public class MacProcesses {

    /**
     * Get a list of all running processes the installer can see.
     * @return info objects with module name, process id and further information
     */
    public static Info[] getRunningProcesses() {
        if (!Util.isMacOS()) {
            return new Info[0];
        }
        Collection<ProcessInfo> processes = RunningProcessChecker.getAllRunningProcesses();
        Info[] infos = new Info[processes.size()];
        int i = 0;
        for (ProcessInfo processInfo : processes) {
            infos[i++] = (Info)processInfo;
        }
        return infos;
    }

    /**
     * Tries to close the processes with the given ids by sending a VM_CLOSE message to all visible top-level windows.
     * @param processIds the processes to terminate
     * @param force if the process should be killed with {@code kill -9}
     * @param timeout the maximum time to wait for the processes to terminate after the message has been sent in milliseconds.
     *                The minimum time is 400 ms regardless of a lower setting.
     * @return {@code true} if all processes have been terminated.
     */
    public static boolean terminateProcesses(int[] processIds, boolean force, int timeout) {
        return MacProcessHelper.terminateProcesses(processIds, force, timeout);
    }

    /**
     * Determine if any of the installed launchers are currently running. All launchers in the currently set installation directory
     * are checked. The "Install Files" action performs this check by default and warns the user that processes are running.
     * With this method, you can check for this condition earlier in your installer.
     * @return {@code true} or {@code false}.
     */
    public static boolean areInstalledLaunchersRunning() {
        if (!InstallerUtil.isMacOS()) {
            return false;
        }
        return RunningProcessChecker.areInstalledLaunchersRunning();
    }

    /**
     * Class that holds information about running macOS processes.
     */
    public static class Info extends ProcessInfo {

        private final String displayedName;
        private final String bundleIdentifier;
        private final boolean backgroundOnly;

        /**
         * Called by the framework.
         */
        public Info(int processId, String moduleName, String displayedName, String bundleIdentifier, boolean backgroundOnly) {
            super(processId, moduleName);
            this.displayedName = displayedName;
            this.bundleIdentifier = bundleIdentifier;
            this.backgroundOnly = backgroundOnly;
        }

        @Override
        public String getWindowTitle() {
            String name = bundleIdentifier == null ? getModuleName() : displayedName;
            int index = name.indexOf(InstallerConstants.APP_CONTENTS_NATIVE_FOLDER);
            if (index > -1) {
                return name.substring(0, index);
            } else {
                return name;
            }
        }

        /**
         * The displayed name of the application.
         * For non-application bundle processes, the simple name of the executable is returned.
         * @return the name
         */
        public String getDisplayedName() {
            return displayedName;
        }

        /**
         * The bundle identifier of the application.
         * @return the bundle identifier, or {@code null} if the process is not associated with an application bundle
         */
        public String getBundleIdentifier() {
            return bundleIdentifier;
        }

        /**
         * Returns if the process is only running in the background and does not display any windows.
         * All non-application bundle processes return {@code true} here, even if they display a GUI by
         * some other means.
         * @return if background only
         */
        public boolean isBackgroundOnly() {
            return backgroundOnly;
        }

        @Override
        public String toString() {
            return "Info{" +
                    "displayedName='" + displayedName + '\'' +
                    ", bundleIdentifier=" + bundleIdentifier +
                    ", backgroundOnly=" + backgroundOnly +
                    ", " + super.toString();
        }
    }

}
