package com.install4j.api.laf;

import com.install4j.api.UiUtil;
import com.install4j.runtime.beans.DynamicLightOrDarkColor;
import com.install4j.runtime.util.ArmedTriStateCheckBox;
import org.jetbrains.annotations.NotNull;

import javax.swing.*;
import java.awt.*;

/**
 * The look and feel enhancer helps screens, form components and styles with specific aspects
 * of creating the UI.
 * <p>
 *  At runtime the current look and feel enhancer can be obtained
 *  from {@link UiUtil#getLookAndFeelEnhancer()}.
 * </p>
 * <p>
 *  For developing your own look and feel, see {@link LookAndFeelHandler}.
 * </p>
 *
 * @author ej-technologies GmbH
 */
public interface LookAndFeelEnhancer {

    /**
     * Default color for the title in alerts boxes. Returned by the default implementation
     * of {@link #getTitleColor()} in light mode.
     */
    Color DEFAULT_LIGHT_TITLE_COLOR = new Color(0, 94, 140);

    /**
     * Returns if the look and feel displays a dark mode.
     * <p>The default value is {@code false}.</p>
     * @see UiUtil#isDarkUi()
     */
    default boolean isDark() {
        return false;
    }

    /**
     * Returns if the look and feel supports switching between light and dark mode
     * via {@link UiUtil#setDarkUi(boolean)}.
     * <p>The default value is {@code false}</p>
     */
    default boolean isDarkModeSwitchingSupported() {
        return false;
    }

    /**
     * Returns if tree selections extend across the entire width of the tree as opposed to
     * just the label.
     * <p>The default value is {@code false}.</p>
     */
    default boolean isWideTreeSelection() {
        return false;
    }

    /**
     * Returns if a component should not be made transparent with a call to
     * {@link JComponent#setOpaque(boolean)} when it is contained in a parent container
     * with a non-default background color.
     * just the label.
     * <p>The default value is {@code false}.</p>
     */
    default boolean isPreventTransparency(JComponent component) {
        return false;
    }

    /**
     * Returns a tri-state checkbox if a specific implementation can be supplied by the look and feel
     * <p>If overridden, it must return an instance that implements
     * {@link IndeterminateStateComponent}.
     * Note that when {@link JCheckBox#setSelected(boolean)} is called,
     * the indeterminate state must be set to false.
     * </p>
     * <p>The default value uses an implementation that permanently activates the armed state for the indeterminate value.</p>
     */
    @NotNull
    default JCheckBox createTriStateCheckBox() {
        return new ArmedTriStateCheckBox();
    }

    /**
     * Returns the color for the title in alerts boxes.
     * <p>The default value is the label font in dark mode and {@link #DEFAULT_LIGHT_TITLE_COLOR}
     * in light mode.
     * </p>
     * <p>Not applicable macOS where native alerts are used.</p>
     */
    default Color getTitleColor() {
        return new DynamicLightOrDarkColor(() -> DEFAULT_LIGHT_TITLE_COLOR, () -> UIManager.getColor("Label.foreground"));
    }

    /**
     * Returns the font for normal text in alert boxes.
     * <p>The default value is the label font.</p>
     * <p>Not applicable macOS where native alerts are used.</p>
     */
    @NotNull
    default Font getAlertFont() {
        return UIManager.getFont("Label.font");
    }

    /**
     * Some look and feels provide a file chooser dialog where the lower accessory should
     * have an inset on the left side.
     * <p>The default value is {@code 5}.</p>
     * <p>Not applicable on Windows and macOS where native file chooser dialogs are used.</p>
     */
    default int getFileChooserLowerAccessoryInset() {
        return 5;
    }

    /**
     * Prepare a window before it is displayable. If a custom look and feel sets its own window decoration, it
     * can make windows undecorated like this:
     * <pre>{@code
     * if (window instanceof Frame) {
     *     ((Frame)window).setUndecorated(true);
     * } else if (window instanceof Dialog) {
     *     ((Dialog)window).setUndecorated(true);
     * }
     * }</pre>
     * <p>
     * The default implementation is empty.
     * </p>
     */
    default void prepareWindow(Window window) {
    }

    /**
     * Advises the look and feel UI not to perform any extra drawing for selected nodes in the specified tree.
     * For example, some look and feels paint a wide selection across the entire width of the tree.
     * This is necessary for trees that handle the painting of the selection themselves.
     */
    default void disableExtraSelectionPainting(JTree tree) {

    }

    /**
     * Returns the custom scale factor for the look and feel.
     * This factor does not include HiDPI scale factors but only an extra scaling of UI element sizes.
     * @return the custom scale factor
     */
    default float getCustomScaleFactor() {
        return 1f;
    }
}
