package com.install4j.api.windows.service;

import com.install4j.api.Util;
import com.install4j.api.context.Context;
import com.install4j.api.windows.WinUser;
import com.install4j.runtime.beans.actions.services.AbstractControlServiceAction;
import com.install4j.runtime.installer.ContextImpl;
import com.install4j.runtime.installer.helper.comm.ExecutionContext;
import com.install4j.runtime.installer.helper.comm.HelperCommunication;
import com.install4j.runtime.installer.platform.win32.Win32Services;
import com.install4j.runtime.installer.platform.win32.Win32UserInfo;
import com.install4j.runtime.launcher.util.LauncherUtil;

/**
 * Collection of static methods to manipulate Windows services
 * @author ej-technologies GmbH
 */
public class WinServices {

    /**
     * Starts a service. To successfully call this method, you have to request privileges first.
     * @param serviceName the name of the service
     * @param parameters optional parameters
     * @throws ServiceException a {@link ServiceNotFoundException} in case serviceName does not refer to the existing service,
     * an {@link ServiceAccessException} if you don't have sufficient privileges or a generic exception with a Win32 error code
     */
    public static void start(final String serviceName, final String... parameters) throws ServiceException {
        checkPrerequisites(serviceName);
        try {
            HelperCommunication.getInstance().fetchObject(ExecutionContext.MAXIMUM, context -> {
                Win32Services.startService(serviceName, parameters);
                return null;
            });
        } catch (RuntimeException e) {
            throw checkServiceException(e);
        }
    }

    /**
     * Stops a service with a default timeout of 30 s.
     * @see #stop(String, int)
     */
    public static void stop(final String serviceName) throws ServiceException {
        Context context = ContextImpl.getCurrentContext();
        final int minimumTimeout = AbstractControlServiceAction.getMinimumWaitTime(context, Win32Services.DEFAULT_MINIMUM_STOP_TIMEOUT);
        stop(serviceName, minimumTimeout);
    }

    /**
     * Stops a service. To successfully call this method, you have to request privileges first.
     * @param serviceName the name of the service
     * @param minimumWaitTime the minimum time in milliseconds that will be waited for the service to shut down.
     * @throws ServiceException a {@link ServiceNotFoundException} in case serviceName does not refer to the existing service,
     * an {@link ServiceAccessException} if you don't have sufficient privileges or a generic exception with a Win32 error code
     */
    public static void stop(final String serviceName, int minimumWaitTime) throws ServiceException {
        checkPrerequisites(serviceName);
        try {
            HelperCommunication.getInstance().fetchObject(ExecutionContext.MAXIMUM, context -> {
                Win32Services.stopService(serviceName, minimumWaitTime);
                return null;
            });
        } catch (RuntimeException e) {
            throw checkServiceException(e);
        }
    }

    /**
     * Installs a service. To successfully call this method, you have to request privileges first.
     * @param serviceName the name of the service
     * @param configuration the configuration of the service. At least the binaryName property must be set.
     * @throws ServiceException an {@link ServiceAccessException} if you don't have sufficient privileges or a generic exception with a Win32 error code
     */
    public static void install(final String serviceName, final ServiceConfiguration configuration) throws ServiceException {
        checkPrerequisites(serviceName);
        if (configuration == null) throw new IllegalArgumentException("configuration required");
        if (configuration.getBinaryName() == null) throw new IllegalArgumentException("configuration.binaryName required");

        try {
            HelperCommunication.getInstance().fetchObject(ExecutionContext.MAXIMUM, context -> {
                Win32Services.installService(serviceName, null, configuration.getBinaryName());
                changeServiceInt(serviceName, configuration);
                return null;
            });
        } catch (RuntimeException e) {
            throw checkServiceException(e);
        }
    }

    /**
     * Changes a service configuration. To successfully call this method, you have to request privileges first.
     * @param serviceName the name of the service
     * @param configuration the configuration of the service. All properties that are not set or set to {@code null} won't be changed.
     * @throws ServiceException a {@link ServiceNotFoundException} in case serviceName does not refer to the existing service,
     * an {@link ServiceAccessException} if you don't have sufficient privileges or a generic exception with a Win32 error code
     */
    public static void change(final String serviceName, final ServiceConfiguration configuration) throws ServiceException {
        checkPrerequisites(serviceName);
        try {
            HelperCommunication.getInstance().fetchObject(ExecutionContext.MAXIMUM, context -> {
                changeServiceInt(serviceName, configuration);
                return null;
            });
        } catch (RuntimeException e) {
            throw checkServiceException(e);
        }
    }

    /**
     * Uninstalls a service. To successfully call this method, you have to request privileges first.
     * @param serviceName the name of the service
     * @throws ServiceException a {@link ServiceNotFoundException} in case serviceName does not refer to the existing service,
     * an {@link ServiceAccessException} if you don't have sufficient privileges or a generic exception with a Win32 error code
     */
    public static void uninstall(final String serviceName) throws ServiceException {
        checkPrerequisites(serviceName);
        try {
            HelperCommunication.getInstance().fetchObject(ExecutionContext.MAXIMUM, context -> {
                Win32Services.uninstallService(serviceName);
                return null;
            });
        } catch (RuntimeException e) {
            throw checkServiceException(e);
        }
    }

    /**
     * Checks if a service is currently running.
     * @param serviceName the name of the service
     * @return {@code true} if running
     * @throws ServiceException a {@link ServiceNotFoundException} in case serviceName does not refer to the existing service,
     * an {@link ServiceAccessException} if you don't have sufficient privileges or a generic exception with a Win32 error code
     */
    public static boolean isRunning(final String serviceName) throws ServiceException {
        checkPrerequisites(serviceName);
        try {
            return HelperCommunication.getInstance().fetchBoolean(ExecutionContext.MAXIMUM, context -> Win32Services.isRunning(serviceName));
        } catch (RuntimeException e) {
            throw checkServiceException(e);
        }
    }

    /**
     * Returns the start type of a service.
     * @param serviceName the name of the service
     * @return a {@link ServiceStartType} value
     * @throws ServiceException a {@link ServiceNotFoundException} in case serviceName does not refer to the existing service,
     * an {@link ServiceAccessException} if you don't have sufficient privileges or a generic exception with a Win32 error code
     */
    public static ServiceStartType getStartType(final String serviceName) throws ServiceException {
        checkPrerequisites(serviceName);
        try {
            return HelperCommunication.getInstance().fetchObject(ExecutionContext.MAXIMUM, context -> {
                int startType = Win32Services.getStartType(serviceName);
                switch (startType) {
                    case Win32Services.SERVICE_AUTO_START:
                        return ServiceStartType.AUTO;
                    case Win32Services.SERVICE_DEMAND_START:
                        return ServiceStartType.DEMAND;
                    case Win32Services.SERVICE_DISABLED:
                        return ServiceStartType.DISABLED;
                    default:
                        throw new RuntimeException("unknown start type " + startType);
                }
            });
        } catch (RuntimeException e) {
            throw checkServiceException(e);
        }
    }

    /**
     * Checks if a service is installed.
     * @param serviceName the name of the service
     * @throws ServiceException a {@link ServiceAccessException} if you don't have sufficient privileges or a generic exception with a Win32 error code
     * @return {@code true} if the service is installed, {@code false} otherwise
     */
    public static boolean isInstalled(String serviceName) {
        try {
            getStartType(serviceName);
            return true;
        } catch (ServiceNotFoundException e) {
            return false;
        }
    }

    /**
     * Returns the registered binary of a service.
     * @param serviceName the name of the service
     * @return the binary name. Can include quotes and arguments if the service was registered like this.
     * @throws ServiceException a {@link ServiceNotFoundException} in case serviceName does not refer to the existing service,
     * an {@link ServiceAccessException} if you don't have sufficient privileges or a generic exception with a Win32 error code
     */
    public static String getBinary(final String serviceName) throws ServiceException {
        checkPrerequisites(serviceName);
        try {
            return HelperCommunication.getInstance().fetchString(ExecutionContext.MAXIMUM, context -> Win32Services.getServiceBinary(serviceName));
        } catch (RuntimeException e) {
            throw checkServiceException(e);
        }
    }

    private static void changeServiceInt(String serviceName, ServiceConfiguration configuration) throws Win32Services.ServiceException {
        int startType = configuration.getStartType() == null ? Win32Services.SERVICE_NO_CHANGE : configuration.getStartType().intValue;
        String accountName = null;
        String password = null;
        if (configuration.getServiceAccount() != null) {
            if (configuration.getServiceAccount() == ServiceAccount.OTHER) {
                accountName = WinUser.getAccountName(configuration.getOtherAccountName());
                if (accountName == null) {
                    accountName = configuration.getOtherAccountName();
                }
                password = configuration.getOtherAccountPassword();
            } else {
                accountName = configuration.getServiceAccount().getAccountName();
                password = "";
            }
        }
        Win32Services.changeServiceConfig(serviceName, configuration.getDisplayName(), configuration.getBinaryName(), false, startType, configuration.getDependencies(), accountName, password, configuration.getDescription());
        if (Util.isAtLeastWindowsVista() && configuration.getDelayedAutoStart() != null) {
            Win32Services.setDelayedAutoStart(serviceName, configuration.getDelayedAutoStart());
        }
        if (configuration.getRestartOnFailure() != null) {
            Win32Services.setRestartServiceConfig(serviceName, configuration.getRestartOnFailure(), configuration.getRestartMillis(), configuration.getMaxRestarts(), configuration.getResetSeconds());
        }
        if (configuration.getServiceAccount() == ServiceAccount.OTHER) {
            Win32UserInfo.setLsaAccountRight(configuration.getOtherAccountName(), Win32UserInfo.RIGHT_SERVICE_LOGON, true);
        }
    }


    private static RuntimeException checkServiceException(RuntimeException e) {
        if (e.getCause() instanceof Win32Services.ServiceException) {
            Win32Services.ServiceException serviceException = (Win32Services.ServiceException)e.getCause();
            switch (serviceException.getErrorCode()) {
                case Win32Services.ERROR_SERVICE_DOES_NOT_EXIST:
                    throw new ServiceNotFoundException();
                case Win32Services.ERROR_ACCESS_DENIED:
                    throw new ServiceAccessException();
                default:
                    throw new ServiceException("Win32 error code: " + serviceException.getErrorCode());
            }
        } else {
            throw e;
        }
    }

    private static void checkPrerequisites(String serviceName) {
        if (!LauncherUtil.isWindows()) throw new ServiceException("only available on Windows");
        if (serviceName == null) throw new IllegalArgumentException("serviceName required");
    }
}
