package com.install4j.api.beaninfo;

import com.install4j.api.beans.ActionList;
import com.install4j.api.beans.Bean;

import java.beans.IntrospectionException;
import java.beans.PropertyChangeListener;
import java.beans.PropertyDescriptor;

/**
 * Property descriptor for use in instances of {@link Install4JBeanInfo}.
 * For file and script properties, derived classes are available.
 * <p>Using this class is not strictly required. In principle, you could also set values for the {@code ATTRIBUTE_*} constants in the property descriptor
 * of an unrelated property descriptor class.
 */
public class Install4JPropertyDescriptor extends PropertyDescriptor {

    /**
     * @see #setPropertyCategory(String)
     */
    public static final String ATTRIBUTE_PROPERTY_CATEGORY = "propertyCategory";

    /**
     * Default category used by install4j.
     */
    public static final String CATEGORY_CONFIGURATION = "Configuration";

    /**
     * @see #create(String, Class, String, String, String)
     */
    public static final String ATTRIBUTE_CONTEXT = "context";

    /**
     * Special context for properties of type {@code java.lang.String} that tells the install4j GUI
     * to treat the string value as the ID of a launcher. In the property panel, a drop-down list of
     * all defined launchers will be shown.
     */
    public static final String CONTEXT_LAUNCHER_ID = "launcher";

    /**
     * Same as {@link #CONTEXT_LAUNCHER_ID}, but with an additional "[Other executable]" entry that sets the id property to null.
     */
    public static final String CONTEXT_LAUNCHER_ID_OR_OTHER = "launcherOrOther";

    /**
     * Same as {@link #CONTEXT_LAUNCHER_ID}, but service launchers are not selectable.
     */
    public static final String CONTEXT_NON_SERVICE_LAUNCHER_ID = "nonServiceLauncher";

    /**
     * Same as {@link #CONTEXT_LAUNCHER_ID}, but only service launchers are selectable.
     */
    public static final String CONTEXT_SERVICE_LAUNCHER_ID = "serviceLauncher";

    /**
     * Special context for properties of type {@code java.lang.String} that tells the install4j GUI
     * to treat the string value as the ID of an installation component. In the property panel, a drop-down list of
     * all defined installation components will be shown.
     */
    public static final String CONTEXT_COMPONENT_ID = "component";

    /**
     * Special context for properties of type {@code java.lang.String} that tells the install4j GUI
     * to treat the string value as the ID of a form component. This is only relevant for properties of form
     * components. In the property panel, a drop-down list of all defined form components in the current screen
     * will be shown. Layout groups and the currently edited form component will not be shown.
     */
    public static final String CONTEXT_FORM_COMPONENT_ID = "formComponent";

    /**
     * Special context for properties of type {@code java.lang.String} that tells the install4j GUI
     * to treat the string value as the ID of a style. This is useful for screen style implementations that need to refer
     * to other user-selectable screen styles, for example, for nesting them. In the property panel, a drop-down list of all
     * defined screen styles will be shown.
     * @see com.install4j.api.styles.Style
     * @see com.install4j.api.styles.StyleManager
     */
    public static final String CONTEXT_STYLE_ID = "style";

    /**
     * Same as {@link #CONTEXT_COMPONENT_ID}, but only downloaded installation components are selectable.
     */
    public static final String CONTEXT_DOWNLOADABLE_COMPONENT_ID = "downloadableComponent";

    /**
     * Special context for properties of type {@code java.lang.String} that tells the install4j GUI
     * to allow the user to enter multi-line strings in a separate dialog.
     */
    public static final String CONTEXT_MULTILINE = "multiline";

    /**
     * Special context for properties of type {@code java.lang.String} that tells the install4j GUI
     * to allow the user to enter an HTML document in a separate editor with syntax coloring and preview
     * functionality.
     */
    public static final String CONTEXT_HTML = "html";

    /**
     * Special context for properties of type {@code java.util.Date} that tells the install4j GUI
     * to offer a time editor besides the default date editor as well.
     */
    public static final String CONTEXT_DATETIME = "dateTime";

    /**
     * Special context for properties of type {@code java.lang.String} that tells the install4j GUI
     * to treat the string value as an installer variable name. The user will be offered a selector to choose
     * pre-defined installer variables.
     * <p>
     * This context can also be applied to properties of type {@code java.lang.String[]} so that the variable selector
     * in the edit dialog inserts installer variable names without the replacement syntax.
     * </p>
     */
    public static final String CONTEXT_VARIABLE_NAME = "variableName";

    /**
     * Special context for properties of type {@code java.lang.String} and {@code java.io.File} that tells the install4j GUI
     * not to offer installer variables in the text field for the property.
     */
    public static final String CONTEXT_NO_INSTALLER_VARIABLES = "noInstallerVariables";

    /**
     * Special context for properties of type {@code java.util.LinkedHashMap} that tells the install4j GUI
     * that the map contains string keys and values and to offer a suitable converter and property
     * customizer.
     */
    public static final String CONTEXT_STRING_TO_STRING_MAP = "stringToStringMap";

    /**
     * Special context for properties of type {@link com.install4j.api.beans.Anchor} that tells the install4j GUI
     * to offer only the corner values of the anchor (north-west, north-east, south-west, south-east).
     */
    public static final String CONTEXT_ANCHOR_CORNERS = "corners";

    /**
     * Special context for properties of type {@link com.install4j.api.beans.Anchor} that tells the install4j GUI
     * to offer only the side values of the anchor (north, west, east, south).
     */
    public static final String CONTEXT_ANCHOR_SIDES = "sides";

    /**
     * Special context for properties of type {@link com.install4j.api.beans.Anchor} that tells the install4j GUI
     * to offer only the horizontal values of the anchor (west, center, east).
     */
    public static final String CONTEXT_ANCHOR_HORIZONTAL = "horizontal";

    /**
     * Special context for properties of type {@link com.install4j.api.beans.Anchor} that tells the install4j GUI
     * to offer only the vertical values of the anchor (north, center, south).
     */
    public static final String CONTEXT_ANCHOR_VERTICAL = "vertical";

    /**
     * Special context for properties of type {@code java.lang.String} that tells the install4j GUI
     * to treat the string value as the ID of a widget style. In the property panel, a drop-down list of
     * all defined widget sets will be shown.
     * <p>
     * As a special value, the empty string denotes the widget style for the current screen style.
     * </p>
     * <p>
     * Widget style IDS are useful if you construct Swing components yourself in code and want to apply a widget style to them.
     * You can pass widget set IDs to the {@code applyStyle...} methods in {@link com.install4j.api.formcomponents.FormEnvironment}
     * to apply the widget style to a Swing component that you have constructed yourself.
     * </p>
     */
    public static final String CONTEXT_WIDGET_STYLE_ID = "widgetStyleId";

    /**
     * @see #setSortKey(Integer)
     */
    public static final String ATTRIBUTE_SORT_KEY = "sortKey";

    /**
     * @see #setPropertyChangeListener(PropertyChangeListener)
     */
    public static final String ATTRIBUTE_PROPERTY_CHANGE_LISTENER = "propertyChangeListener";

    /**
     * @see #setParentProperty(String)
     */
    public static final String ATTRIBUTE_PARENT_PROPERTY = "parentProperty";

    /**
     * @see #setVariableValueClass(Class)
     */
    public static final String ATTRIBUTE_VARIABLE_VALUE_CLASS = "variableValueClass";

    /**
     * @see #setVisibilityDiscriminator(VisibilityDiscriminator)
     */
    public static final String ATTRIBUTE_VISIBILITY_DISCRIMINATOR = "visibilityDiscriminator";

    /**
     * @see #setActionListHiddenKeys(String[])
     */
    public static final String ATTRIBUTE_ACTION_LIST_HIDDEN_KEYS = "actionListHiddenKeys";

    /**
     * @see #setActionListShownKeys(String[])
     */
    public static final String ATTRIBUTE_ACTION_LIST_SHOWN_KEYS = "actionListShownKeys";

    /**
     * @see #setAllowTextOverride(boolean)
     */
    public static final String ATTRIBUTE_ALLOW_TEXT_OVERRIDE = "allowTextOverride";

    /**
     * @see #setEmptyMessage(String)
     */
    public static final String ATTRIBUTE_EMPTY_MESSAGE = "emptyMessage";

    /**
     * @see #setSuggestedValues(String[])
     */
    public static final String ATTRIBUTE_SUGGESTED_VALUES = "suggestedValues";

    /**
     * @see #setNestedBean(boolean)
     */
    public static final String ATTRIBUTE_NESTED_BEAN = "nestedBean";

    /**
     * @see #setNestedBeanCategory(String)
     */
    public static final String ATTRIBUTE_NESTED_BEAN_CATEGORY = "nestedBeanCategory";


    /**
     * @see #setNestedBeanPropertyFilter(PropertyFilter)
     */
    public static final String ATTRIBUTE_NESTED_BEAN_PROPERTY_FILTER = "nestedBeanPropertyFilter";


    /**
     * Same as {@link #create(String, Class, String, String, String)} with a {@code context} of {@code null}.
     */
    public static Install4JPropertyDescriptor create(String propertyName, Class beanClass, String displayName, String shortDescription) {
        return create(propertyName, beanClass, displayName, shortDescription, null);
    }

    /**
     * Create a property descriptor that can be passed to {@link Install4JBeanInfo#addPropertyDescriptor(Install4JPropertyDescriptor)}.
     * @param propertyName the name of the property
     * @param beanClass the class of the bean that contains the property. <b>Note:</b> This is <b>not</b> the class of the property.
     * @param displayName the display name of the property
     * @param shortDescription a short description of the property in HTML format. You do not have to start the description with &lt;html&gt;, it will be prepended automatically.
     * @param context the context of the property. The context allows specifying different editors for the same property types.
     * See the {@code CONTEXT_*} properties for the built-in contexts. This context is also used for enumerated properties
     * in the {@link EnumerationMapper}. For properties of type {@code String[]}, you can pass an item name as the context.
     * @return the property descriptor
     */
    public static Install4JPropertyDescriptor create(String propertyName, Class beanClass, String displayName, String shortDescription, String context) {
        try {
            Install4JPropertyDescriptor descriptor = createDescriptor(propertyName, beanClass);
            descriptor.setDisplayName(displayName);
            descriptor.setShortDescription(shortDescription);
            descriptor.setContext(context);
            return descriptor;
        } catch (IntrospectionException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }


    /**
     * Same as {@link #createNested(String, String, Class)} with a {@code nestedBeanCategory} of {@code null}.
     */
    public static Install4JPropertyDescriptor createNested(String propertyName, Class beanClass) {
        return createNested(propertyName, null, beanClass);
    }

    /**
     * Same as {@link #createNested(String, String, Class, PropertyFilter)} with a {@code propertyFilter} of {@code name -> true}.
     */
    public static Install4JPropertyDescriptor createNested(String propertyName, String nestedBeanCategory, Class beanClass) {
        return createNested(propertyName, nestedBeanCategory, beanClass, name -> true);
    }

    /**
     * Create a property descriptor for a nested bean that can be passed to {@link Install4JBeanInfo#addPropertyDescriptor(Install4JPropertyDescriptor)}.
     * <p>
     * Note that nested bean values must always be non-null.
     * </p>
     *
     * @param propertyName       the name of the property
     * @param nestedBeanCategory the category in which the properties of the nested bean will be shown
     * @param beanClass          the class of the bean that contains the property. <b>Note:</b> This is <b>not</b> the class of the property.
     * @return the property descriptor
     * @see #setNestedBean(boolean)
     */
    public static Install4JPropertyDescriptor createNested(String propertyName, String nestedBeanCategory, Class beanClass, PropertyFilter propertyFilter) {
        try {
            Install4JPropertyDescriptor descriptor = createDescriptor(propertyName, beanClass);
            descriptor.setNestedBean(true);
            descriptor.setNestedBeanCategory(nestedBeanCategory);
            descriptor.setNestedBeanPropertyFilter(propertyFilter);
            return descriptor;
        } catch (IntrospectionException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    private static Install4JPropertyDescriptor createDescriptor(String propertyName, Class beanClass) throws IntrospectionException {
        try {
            return new Install4JPropertyDescriptor(propertyName, beanClass);
        } catch (Exception ignored) {
        }
        try {
            return new Install4JPropertyDescriptor(propertyName, beanClass, "is" + capitalize(propertyName), null);
        } catch (Exception ignored) {
        }
        try {
            return new Install4JPropertyDescriptor(propertyName, beanClass, "get" + capitalize(propertyName), null);
        } catch (IntrospectionException e) {
            e.printStackTrace();
            throw e;
        }

    }

    static String capitalize(String name) {
        if (name == null || name.isEmpty()) {
            return name;
        }
        return name.substring(0, 1).toUpperCase() + name.substring(1);
    }

    Install4JPropertyDescriptor(String propertyName, Class beanClass) throws IntrospectionException {
        super(propertyName, beanClass);
    }

    Install4JPropertyDescriptor(String propertyName, Class beanClass, String readMethod, String writeMethod) throws IntrospectionException {
        super(propertyName, beanClass, readMethod, writeMethod);
    }

    /**
     * Set the context for the property.
     * @see #create(String, Class, String, String, String) for an explanation of the context.
     * @param editorContext the context
     * @return {@code this}, for chained calls on this property descriptor
     */
    public Install4JPropertyDescriptor setContext(String editorContext) {
        if (editorContext != null) {
            setValue(ATTRIBUTE_CONTEXT, editorContext);
        }
        return this;
    }

    /**
     * Configures the property category. In the install4j GUI, properties are grouped by their categories. If
     * no category has been specified, the default "Configuration" category is used.
     * <p>You can collapse certain categories by default with the
     * {@link Install4JBeanInfo#setCollapsedPropertyCategories(String[])} method.
     * @param propertyCategory the category
     * @return {@code this}, for chained calls on this property descriptor
     */
    public Install4JPropertyDescriptor setPropertyCategory(String propertyCategory) {
        if (propertyCategory != null) {
            setValue(ATTRIBUTE_PROPERTY_CATEGORY, propertyCategory);
        }
        return this;
    }

    @Override
    public void setDisplayName(String displayName) {
        super.setDisplayName(displayName);
        if (displayName == null || displayName.trim().isEmpty()) {
            setHidden(true);
        }
    }

    /**
     * Set an integer key by which properties are sorted. If no sort keys are specified, the properties will be sorted
     * in the order they were added to the bean info.
     * @param sortKey the sort key
     * @return {@code this}, for chained calls on this property descriptor
     * @see Install4JBeanInfo#addPropertyDescriptor(Install4JPropertyDescriptor)
     */
    public Install4JPropertyDescriptor setSortKey(Integer sortKey) {
        if (sortKey != null) {
            setValue(ATTRIBUTE_SORT_KEY, sortKey);
        }
        return this;
    }

    /**
     * Set the class of the variable value, if {@link #CONTEXT_VARIABLE_NAME} is set.
     * This information is displayed in the install4j IDE.
     * @param variableClass the class of the variable value
     * @return {@code this}, for chained calls on this property descriptor
     */
    public Install4JPropertyDescriptor setVariableValueClass(Class variableClass) {
        if (variableClass != null) {
            setValue(ATTRIBUTE_VARIABLE_VALUE_CLASS, variableClass);
        }
        return this;
    }

    /**
     * Set a property change listener that is invoked when the property is changed in the IDE. The source of
     * the change event is the bean itself.
     * @param listener the listener
     * @return {@code this}, for chained calls on this property descriptor
     */
    public Install4JPropertyDescriptor setPropertyChangeListener(PropertyChangeListener listener) {
        if (listener != null) {
            setValue(ATTRIBUTE_PROPERTY_CHANGE_LISTENER, listener);
        }
        return this;
    }

    /**
     * Set the parent property under which this property should be shown in the install4j IDE.
     * The property sheet is a tree table, and each property can have child nodes. By assigning parent
     * properties, you can build up a tree-like structure in the property sheet.
     * <p>
     * For boolean properties, only descendant properties are only shown if the boolean value of the property
     * is selected as {@code true}. If the user selection is false, all descendant properties will be hidden.
     * If the selection is changed to true, all descendant properties will be shown.
     * </p>
     * @param parentProperty the name of the parent property
     * @return {@code this}, for chained calls on this property descriptor
     */
    public Install4JPropertyDescriptor setParentProperty(String parentProperty) {
        if (parentProperty != null) {
            setValue(ATTRIBUTE_PARENT_PROPERTY, parentProperty);
        }
        return this;
    }

    /**
     * Sets a visibility discriminator for this property. This only makes sense if
     * {@link #setParentProperty(String)} has been called for this property descriptor.
     * Note that for a boolean parent property, a default visibility discriminator is
     * set that hides children if the user selection of the parent property is {@code false}.
     * @param visibilityDiscriminator the visibility discriminator
     * @return {@code this}, for chained calls on this property descriptor
     * @see VisibilityDiscriminator
     */
    public Install4JPropertyDescriptor setVisibilityDiscriminator(VisibilityDiscriminator visibilityDiscriminator) {
        if (visibilityDiscriminator != null) {
            setValue(ATTRIBUTE_VISIBILITY_DISCRIMINATOR, visibilityDiscriminator);
        }
        return this;
    }

    /**
     * Hide this property if the bean is shown in the configuration of an action list property whose configured action keys
     * contain one of the specified strings.
     * @param hiddenKeys an array with the contexts of the {@link ActionList} property for which the property should be hidden.
     *                       If one of the array elements is {@code null}, the property is always hidden in action lists.
     * @return {@code this}, for chained calls on this property descriptor
     * @see ActionListPropertyDescriptor#setActionKeys(String[])
     */
    public Install4JPropertyDescriptor setActionListHiddenKeys(String[] hiddenKeys) {
        if (hiddenKeys != null) {
            setValue(ATTRIBUTE_ACTION_LIST_HIDDEN_KEYS, hiddenKeys);
        }
        return this;
    }

    /**
     * Show this property only if the bean is shown in the configuration of an action list property whose configured action keys
     * contain one of the specified strings.
     * @param shownKeys an array with the contexts of the {@link ActionList} property for which the property should be shown.
     * @return {@code this}, for chained calls on this property descriptor
     * @see ActionListPropertyDescriptor#setActionKeys(String[])
     */
    public Install4JPropertyDescriptor setActionListShownKeys(String[] shownKeys) {
        if (shownKeys != null) {
            setValue(ATTRIBUTE_ACTION_LIST_SHOWN_KEYS, shownKeys);
        }
        return this;
    }

    /**
     * Do not show this property if the bean is shown in the configuration of the action list.
     * Equivalent to a call to {@code setActionListHiddenContexts(new String[] {null})}
     * @return {@code this}, for chained calls on this property descriptor
     * @see ActionListPropertyDescriptor
     */
    public Install4JPropertyDescriptor setHiddenInActionLists() {
        setActionListHiddenKeys(new String[] {null});
        return this;
    }

    /**
     * Allow the user to switch to text mode in the context menu when editing the property in the install4j IDE.
     * This only has an effect for non-String properties. In the getter of the property, you have to call one of the
     * {@code replaceWithTextOverride} methods in {@link com.install4j.api.beans.AbstractBean}
     * @param allowTextOverride if text mode should be available
     * @return {@code this}, for chained calls on this property descriptor
     * @see com.install4j.api.beans.AbstractBean#getTextOverrideValue(Bean, String, Class)
     */
    public Install4JPropertyDescriptor setAllowTextOverride(boolean allowTextOverride) {
        setValue(ATTRIBUTE_ALLOW_TEXT_OVERRIDE, allowTextOverride);
        return this;
    }

    /**
     * If the value is {@code null} or empty, display a special text in a shaded color in the install4j IDE.
     * This only has an effect for properties of type {@code java.lang.String}, {@code java.lang.String[]} and {@code java.io.File}.
     * @param message the message
     * @return {@code this}, for chained calls on this property descriptor
     */
    public Install4JPropertyDescriptor setEmptyMessage(String message) {
        setValue(ATTRIBUTE_EMPTY_MESSAGE, message);
        return this;
    }

    /**
     * Enable auto-completion of a list of values when using the text field property editor in the install4j IDE.
     * This only has an effect for properties of type {@code java.lang.String}.
     * @param values the values that should be auto-completed.
     * @return {@code this}, for chained calls on this property descriptor
     */
    public Install4JPropertyDescriptor setSuggestedValues(String[] values) {
        setValue(ATTRIBUTE_SUGGESTED_VALUES, values);
        return this;
    }

    /**
     * Mark a property as a nested bean. The type of the property must be a bean type, i.e., a class
     * with a BeanInfo of its own. In the install4j IDE, the properties of that type will be expanded
     * inline in the property table.
     * <p>{@link #setVisibilityDiscriminator(VisibilityDiscriminator) Visibility discriminator}, {@link #setPropertyCategory(String) property category}
     * and {@link #setParentProperty(String) parent property} that are configured for this instance, are applied to
     * all properties of the nested bean. Nesting works recursively.</p>
     * @param nestedBean if the property is a nested bean
     * @return {@code this}, for chained calls on this property descriptor
     */
    public Install4JPropertyDescriptor setNestedBean(boolean nestedBean) {
        setValue(ATTRIBUTE_NESTED_BEAN, nestedBean);
        return this;
    }

    /**
     * Set the category for properties of the nested bean.
     * Only relevant if {@link #setNestedBean(boolean)} has been called with {@code true}.
     * @param category the category in which the properties of the nested bean will be shown
     * @return {@code this}, for chained calls on this property descriptor
     */
    public Install4JPropertyDescriptor setNestedBeanCategory(String category) {
        if (category != null) {
            setValue(ATTRIBUTE_NESTED_BEAN_CATEGORY, category);
        }
        return this;
    }

    /**
     * Set a propertyFilter for properties of the nested bean.
     * Only relevant if {@link #setNestedBean(boolean)} has been called with {@code true}.
     * @param propertyFilter only show properties of the nested bean for which the propertyFilter returns {@code true}.
     * @return {@code this}, for chained calls on this property descriptor
     */
    public Install4JPropertyDescriptor setNestedBeanPropertyFilter(PropertyFilter propertyFilter) {
        if (propertyFilter != null) {
            setValue(ATTRIBUTE_NESTED_BEAN_PROPERTY_FILTER, propertyFilter);
        }
        return this;
    }

    /**
     * Highlight a property as important.
     * If set, the property name is shown in bold font in the IDE.
     * @param preferred whether to show the property in bold font or not.
     */
    @Override
    public void setPreferred(boolean preferred) {
        super.setPreferred(preferred);
    }
}
