package com.install4j.api.windows;

import com.install4j.api.ProcessInfo;
import com.install4j.api.Util;
import com.install4j.runtime.installer.helper.InstallerUtil;
import com.install4j.runtime.installer.helper.RunningProcessChecker;
import com.install4j.runtime.installer.platform.win32.Misc;

import java.util.Collection;

/**
 * Collection of static methods to check for running processes on Windows and terminate them.
 * @author ej-technologies GmbH
 */
public class WinProcesses {

    /**
     * Get a list of all running processes the installer can see.
     * @return info objects with module name and process id.
     */
    public static Info[] getRunningProcesses() {
        if (!Util.isWindows()) {
            return new Info[0];
        }
        Collection<ProcessInfo> processes = RunningProcessChecker.getAllRunningProcesses();
        Info[] infos = new Info[processes.size()];
        int i = 0;
        for (ProcessInfo processInfo : processes) {
            infos[i++] = (Info)processInfo;
        }
        return infos;
    }

    /**
     * Determine if any of the installed launchers are currently running. All launchers in the currently set installation directory
     * are checked. The "Install Files" action performs this check by default and warns the user that processes are running.
     * With this method, you can check for this condition earlier in your installer.
     * @return {@code true} or {@code false}.
     */
    public static boolean areInstalledLaunchersRunning() {
        if (!InstallerUtil.isWindows()) {
            return false;
        }
        return RunningProcessChecker.areInstalledLaunchersRunning();
    }

    /**
     * Get the title of one of the visible top-level windows of this process.
     * @param processId the process id
     * @return the title
     */
    public static String getWindowTitle(int processId) {
        return Misc.getWindowTitle(processId);
    }

    /**
     * Terminates the processes with the given ids forcefully.
     * @param processIds the processes to terminate
     * @return {@code true} if all processes have been terminated.
     */
    public static boolean terminateProcesses(int[] processIds) {
        return Misc.terminateProcesses(processIds, true, 0) == Misc.TERMINATE_SUCCESS;
    }

    /**
     * Tries to close the processes with the given ids by sending a WM_CLOSE message to all visible top-level windows.
     * @param processIds the processes to terminate
     * @param timeout the maximum time to wait for the processes to terminate after the message has been sent in milliseconds
     * @return {@code true} if all processes have been terminated.
     */
    public static boolean closeProcesses(int[] processIds, int timeout) {
        return Misc.terminateProcesses(processIds, false, timeout) == Misc.TERMINATE_SUCCESS;
    }


    /**
     * Class that holds information about running Windows processes.
     */
    public static class Info extends ProcessInfo {
        /**
         * Called by the framework.
         */
        public Info(int processId, String moduleName) {
            super(processId, moduleName);
        }

        @Override
        public String getWindowTitle() {
            return WinProcesses.getWindowTitle(getProcessId());
        }

        @Override
        public String toString() {
            return "Info{" + super.toString();
        }

    }
}

