package com.install4j.api.update;

import java.io.Serializable;

/**
 * Contains information about available updates.
 * <p>
 * Please see the "Auto-Update" help topic in the documentation for a detailed discussion on updaters.
 * <p>
 * When building the installers, install4j creates a file {@code updates.xml} in the media file output directory.
 * The data in that file is exposed by this class. When you place the {@code updates.xml} on a web server,
 * the "Check for update" action can download it and save an instance of this class to an installer variable.
 * You can also download an update descriptor programmatically in your application by using the
 * {@link UpdateChecker} class.
 *
 * @author ej-technologies GmbH
 */
public interface UpdateDescriptor extends Serializable {
    /**
     * Returns all descriptor entries for the different media files of the available version.
     * @return the descriptor entries
     */
    UpdateDescriptorEntry[] getEntries();

    /**
     * An entry that has been identified as a match for the current installation and should be used for
     * choosing an update installer.
     * <p>
     * This is only non-null if the following conditions are fulfilled:
     * <ul>
     * <li> there is indeed a newer version available
     * <li> the conditions for the maximum and minimum updatable version are satisfied by the current installation
     * <li> an entry is available whose target media set ID is equal to the media file ID of the current installation.
     * Media file IDs can be shown in the media section by invoking "Media-&gt;Show IDs" from the
     * main menu.
     * </ul>
     * @return the descriptor entry suitable for updating or {@code null} if no such entry can be found
     */
    UpdateDescriptorEntry getPossibleUpdateEntry();

    /**
     * Return an entry that has the same media file ID as the current installation.
     * <p>
     * This is useful if the logic in {@link #getPossibleUpdateEntry()} is not what you are looking for and
     * you need to implement your own logic while retaining the match in the media file ID.
     * No version checks will be performed for the returned entry. If {@link #getPossibleUpdateEntry()} return
     * a non-null value, this method will return the same value.
     *
     * @return the descriptor entries or {@code null} if no entry with the same media file ID can be found
     * @see #getPossibleUpdateEntry()
     */
    UpdateDescriptorEntry getEntryForCurrentMediaFileId();

    /**
     * Returns the base URL from which update installers will be downloaded.
     * <p>
     * The base URL determines the value of {@link UpdateDescriptorEntry#getURL()} which
     * returns the URL an update installer will be downloaded from. By default, the base URL is the empty string,
     * and the update installers will be downloaded from the same directory as the {@code updates.xml} file.
     * If {@code updates.xml} and the update installers are not placed into
     * the same directory on the web server, you have to change the {@code baseUrl} attribute in the {@code updateDescriptor}
     * element in {@code updates.xml} to the appropriate URL.
     * @return the base URL
     */
    String getBaseUrl();
}
