package com.install4j.api.context;

import com.install4j.api.beans.Anchor;

import java.awt.*;
import java.io.File;

/**
 * A wizard index is used to configure a display of overall installation progress on the left side of the wizard.
 * Note that you can configure the wizard index on each screen in the install4j IDE. For more complex scenarios, this API
 * is provided.
 * <p>By default, no wizard index is set. You can set a wizard index by calling
 * {@link WizardContext#setWizardIndex(WizardIndex)} in a script.
 * If you install the wizard context in the "Pre-activation script" property, it will be used for the current screen.
 * When installed in any other scripts, set the wizard screen will become active once the next screen is shown.
 * <p>The current wizard index is persistent, meaning that it will remain in place for all subsequent screens until the index
 * key of the wizard is changed or the wizard index is set to {@code null} again. A simple usage scenario is given below:
 * <ol>
 * <li>Install the * wizard index on the first screen by invoking
 * <pre>{@code
 * context.getWizardContext().setWizardIndex(new WizardIndex(new String[]
 *      {"Gather Requirements", "Installation", "Service Setup"}));
 * }</pre>
 * in the "Pre-activation" script of that screen.
 * When the wizard context is set that way, the keys for the index steps are auto-generated and set to "1", "2" and "3".
 * If they should be mnemonic, use the 2-arg constructor where you can specify an array with keys as the second argument.
 * When you install the wizard index, the first step is highlighted automatically.
 * <li>For each screen, where the highlighted index step should change, call
 * <pre>{@code
 * context.getWizardContext().setWizardIndexKey("2");
 * }</pre>
 * to highlight step 2 ("Installation") and
 * <pre>{@code
 * context.getWizardContext().setWizardIndexKey("3");
 * }</pre>
 * to highlight step 3 ("Service setup").
 * </ol>
 * <p>Advanced tasks:
 * <ul>
 * <li>To remove the wizard call
 * <pre>{@code
 * context.getWizardContext().setWizardIndex(null);
 * }</pre>
 * <li> You can set another wizard index at any time. When the user goes back in the history, the last shown wizard index and the
 * selected step are shown for each screen.
 * <li>The width of the index panel depends on its contents. However, there are configurable minimum and maximum sizes. The default
 * values are 120 and 300 pixels, respectively. To change these values, use the {@code minWidth} and {@code maxWidth} methods:
 * <pre>{@code
 * new WizardIndex(new String[] {...}}.maxWidth(400).minWidth(50)
 * }</pre>
 * <li>A wizard index can be <b>partially defined</b>. In that case, a "..." entry will be shown at the bottom. This
 * is useful if the user makes a selection that will determine the contents of the full wizard index. For example, on the first
 * screen, call
 * <pre>{@code
 * context.getWizardContext().setWizardIndex(new WizardIndex(
 *      new String[] {"Installation type"})).partiallyDefined(true);
  * }</pre>
 * In the "Validation expression" of the screen, you then skip to two different sequences of screens, depending on the selected
 * installation type. In the first screens of each sequence, install new wizard indices with their full respective contents, but
 * repeat the "Installation type" entry and set the selected wizard index key to the second element immediately. In this way, #
 * the partially defined wizard index expands to the full content on the second screen.
 * <li>You can customize the background of the wizard index by setting
 * <ul>
 * <li>the foreground color used for text with {@code foreground(Color)}
 * <li>the background color with {@code background(Color)}
 * <li>the background image with {@code backgroundImage(File)}
 * <li>the anchor of the background image with {@code backgroundImageAnchor(anchor)}
 * </ul>
 * </ul>
 *
 * @author ej-technologies GmbH
 */

public class WizardIndex {

    private String[] stepNames;
    private String[] keys;
    private boolean partiallyDefined;
    private int maxWidth = 300;
    private int minWidth = 120;
    private boolean numbered = true;
    private Color background;
    private Color foreground;
    private File backgroundImage;
    private Anchor backgroundImageAnchor = Anchor.SOUTHWEST;
    private String initialKey;

    /**
     * Construct a wizard index with the specified step names and auto-generate the index keys to the strings "1", "2", "3", and so on.
     * @param stepNames the displayed step names
     */
    public WizardIndex(String[] stepNames) {
        this(stepNames, null);
    }

    /**
     * Construct a wizard index with the specified step names and index keys.
     * @param stepNames the displayed step names
     * @param keys the keys for the step names, to be used in {@link WizardContext#setWizardIndexKey(String)}
     */
    public WizardIndex(String[] stepNames, String[] keys) {
        if (stepNames == null || stepNames.length == 0) {
            throw new IllegalArgumentException("step names must have a length > 0");
        }
        this.stepNames = stepNames;
        if (keys == null) {
            keys = createKeys(stepNames);
        }
        if (keys.length != stepNames.length) {
            throw new IllegalArgumentException("step names and keys must have the same lengths");
        }
        this.keys = keys;
    }

    /**
     * Returns the step names
     * @return the step names
     */
    public String[] getStepNames() {
        return stepNames;
    }

    /**
     * Returns the index keys
     * @return the index keys
     */
    public String[] getKeys() {
        return keys;
    }

    /**
     * Returns if the wizard index is partially defined.
     * @return if partially defined
     */
    public boolean isPartiallyDefined() {
        return partiallyDefined;
    }

    /**
     * Sets if the wizard index should be partially defined. In this case, a "..." entry will be appended
     * to the end of the wizard index. By default, this is set to {@code false}.
     * @param partiallyDefined if partially defined
     * @return this object, for use in a builder pattern
     */
    public WizardIndex partiallyDefined(boolean partiallyDefined) {
        this.partiallyDefined = partiallyDefined;
        return this;
    }

    /**
     * Returns the maximum width of the index panel
     * @return the maximum width in pixels
     */
    public int getMaxWidth() {
        return maxWidth;
    }

    /**
     * Sets the maximum width of the index panel. By default, this is set to 300 pixels.
     * @param maxWidth the maximum width in pixels
     * @return this object, for use in a builder pattern
     */
    public WizardIndex maxWidth(int maxWidth) {
        this.maxWidth = maxWidth;
        return this;
    }

    /**
     * Returns the minimum width of the index panel.
     * @return the minimum width in pixels
     */
    public int getMinWidth() {
        return minWidth;
    }

    /**
     * Sets the minimum width of the index panel. By default, this is set to 120 pixels.
     * @param minWidth the minimum width
     * @return this object, for use in a builder pattern
     */
    public WizardIndex minWidth(int minWidth) {
        this.minWidth = minWidth;
        return this;
    }

    /**
     * Returns if numbers should be added in front of each index step.
     * @return {@code true} or {@code false}
     */
    public boolean isNumbered() {
        return numbered;
    }

    /**
     * Sets if numbers should be added in front of each index step. By default, this is
     * set to {@code true}.
     * @param numbered  {@code true} or {@code false}
     * @return this object, for use in a builder pattern
     */
    public WizardIndex numbered(boolean numbered) {
        this.numbered = numbered;
        return this;
    }

    /**
     * Returns the background color of the index panel
     * @return the background color or {@code null} if no background color has been set
     */
    public Color getBackground() {
        return background;
    }

    /**
     * Sets the background color of the index panel. The default background color is white.
     * @param background the background color
     * @return this object, for use in a builder pattern
     */
    public WizardIndex background(Color background) {
        this.background = background;
        return this;
    }

    /**
     * Returns the foreground color of the index panel used for the text color
     * @return the foreground color or {@code null} if no foreground color has been set
     */
    public Color getForeground() {
        return foreground;
    }

    /**
     * Sets the foreground color of the index panel. The default foreground color is the foreground colors of labels.
     * @param foreground the foreground color
     * @return this object, for use in a builder pattern
     */
    public WizardIndex foreground(Color foreground) {
        this.foreground = foreground;
        return this;
    }

    /**
     * Returns the background image file.
     * @return the background image file or {@code null} if no background image file has been set.
     */
    public File getBackgroundImage() {
        return backgroundImage;
    }

    /**
     * Sets the background image file. A simple file name without a prefixed path like {@code new File("image.png")}
     * is resolved to the resource directory. You can add image files on the Installer-&gt;Custom Code &amp;Resources tab
     * and use them that way.
     * @param backgroundImage the image file
     * @return this object, for use in a builder pattern
     */
    public WizardIndex backgroundImage(File backgroundImage) {
        this.backgroundImage = backgroundImage;
        return this;
    }

    /**
     * Returns the anchor for the background image file.
     * @return the anchor
     */
    public Anchor getBackgroundImageAnchor() {
        return backgroundImageAnchor;
    }

    /**
     * Sets the anchor for the background image file. The default anchor is SOUTH_WEST.
     * @param anchor the anchor
     * @return this object, for use in a builder pattern
     */
    public WizardIndex backgroundImageAnchor(Anchor anchor) {
        backgroundImageAnchor = anchor;
        return this;
    }

    /**
     * Returns the key of the step name that should be initially selected
     * @return the key.
     */
    public String getInitialKey() {
        return initialKey == null ? keys[0] : initialKey;
    }

    /**
     * Sets the key of the step name that should be initially selected. By default, the first step will be selected.
     * @param initialKey the key
     * @return this object, for use in a builder pattern
     */
    public WizardIndex initialKey(String initialKey) {
        this.initialKey = initialKey;
        return this;
    }

    private String[] createKeys(String[] stepNames) {
        String[] keys = new String[stepNames.length];
        for (int i = 0; i < stepNames.length; i++) {
            keys[i] = String.valueOf(i + 1);
        }
        return keys;
    }

}
