package com.install4j.jdk.spi;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * A node that describes a JDK release and its download URLs.
 * For the arguments named {@code platform}, the platform IDs in {@link CommonPlatforms} must be used when available.
 * Other platforms can use arbitrary IDs.
 */
public interface JdkReleaseNode extends JdkNode {
    /**
     * Returns the download URL for a given platform. May be null if this platform is not available in this release.
     */
    @Nullable
    String getDownloadUrl(String platform);

    /**
     * Returns the optional separate jmods download URL for a given platform. May be null if this platform is not available in this release
     * or if there is no separate jmods download for this platform.
     */
    @Nullable
    default String getJmodsUrl(String platform) {
        return null;
    }

    /**
     * Returns the file name that should be used for the generated JRE bundle for a given platform.
     */
    @Nullable
    String getFileName(@NotNull String platform);

    /**
     * Returns the config key for this release. The config key is saved in the install4j project file and
     * passed to {@link JdkProvider#getByConfigKey(String)} to find a release node when the JDK is downloaded.
     * <p>
     * The config key must start with the Java major version number followed by other characters, for example
     *  {@code 11/11.0.9-b74+1} or {@code 11 FX/11.0.9-b74+1}.
     * </p>
     */
    @NotNull
    String getConfigKey();

    /**
     * Returns the IDs of all available platforms for this release.
     */
    @NotNull
    List<String> getPlatforms();

    /**
     * The java version in the format of the "java.version" system property.
     */
    @NotNull
    String getJavaVersion();
}
