package com.install4j.api.styles;

import com.install4j.api.context.ControlButtonType;

import javax.swing.*;
import java.util.Collection;
import java.util.Collections;

/**
 * Abstract base class for a screen style that wraps a single user-selectable style.
 * <p>
 * When you use this base class, you do not have to handle any life-cycle aspects of the nested style.
 * See {@link NestedStyleContainer} for more information on nesting styles in general.
 * </p>
 * <p>The "customCode" sample project contains the class {@code SunnySkyBackgroundStyle} and its associated
 * {@code BeanInfo} that show you an example of wrapping a user-selected screen style with a graphical decoration.
 * For more complex use cases, the source code of {@link WrapperStyle} can be used as a starting point.
 * </p>
 *
 * @author ej-technologies GmbH
 */
public abstract class WrapperStyle extends AbstractStyle implements NestedStyleContainer {

    private Style nestedStyle;

    /**
     * Returns the ID of the nested style. This method must be implemented by derived classes. To make a
     * {@code String} property user-selectable in the IDE, register it in the {@code BeanInfo} with the special context
     * {@link com.install4j.api.beaninfo.Install4JPropertyDescriptor#CONTEXT_STYLE_ID}.
     *
     * @return the ID
     */
    protected abstract String getNestedStyleId();

    /**
     * Create the UI component for the style. Has the same responsibility as
     * {@link #createComponent()}, only that the UI component of the nested screen style is passed as a parameter.
     *
     * @return the UI component
     */
    protected abstract JComponent createComponent(JComponent styleComponent);

    @Override
    public void setStyleContext(StyleContext styleContext) {
        initStyle();
        if (nestedStyle != null) {
            nestedStyle.setStyleContext(styleContext);
        }
    }

    @Override
    public void setControlButtonEnabled(ControlButtonType controlButtonType, boolean enabled) {
        nestedStyle.setControlButtonEnabled(controlButtonType, enabled);
    }

    @Override
    public void setControlButtonVisible(ControlButtonType controlButtonType, boolean visible) {
        nestedStyle.setControlButtonVisible(controlButtonType, visible);
    }

    @Override
    public void setControlButtonText(ControlButtonType controlButtonType, String text) {
        nestedStyle.setControlButtonText(controlButtonType, text);
    }

    @Override
    public void focusControlButton(ControlButtonType controlButtonType) {
        nestedStyle.focusControlButton(controlButtonType);
    }

    @Override
    public final JComponent createComponent() {
        initStyle();
        return createComponent(getStyleManager().createStyleComponent(nestedStyle));
    }

    @Override
    public boolean isFillHorizontal() {
        if (nestedStyle != null) {
            return nestedStyle.isFillHorizontal();
        } else {
            return false;
        }
    }

    @Override
    public boolean isFillVertical() {
        if (nestedStyle != null) {
            return nestedStyle.isFillVertical();
        } else {
            return false;
        }
    }

    @Override
    public Collection<String> getNestedStyleIds() {
        return Collections.singleton(getNestedStyleId());
    }

    @Override
    public void willActivate() {
        super.willActivate();
        getStyleManager().willActivate(nestedStyle);
    }

    @Override
    public void activated() {
        super.activated();
        getStyleManager().activated(nestedStyle);
    }

    @Override
    public void deactivated() {
        super.deactivated();
        getStyleManager().deactivated(nestedStyle);
    }

    private void initStyle() {
        if (nestedStyle == null) {
            nestedStyle = getStyleManager().cloneStyleById(getNestedStyleId());
        }
    }
}
