package com.install4j.api.context;

import com.install4j.api.actions.Action;
import com.install4j.api.beans.Bean;
import com.install4j.api.beans.ExternalFile;
import com.install4j.api.beans.LocalizedExternalFile;
import com.install4j.api.beans.ScriptProperty;
import com.install4j.api.events.InstallerEventListener;
import com.install4j.api.screens.Screen;

import java.io.File;
import java.io.Serializable;
import java.util.Collection;
import java.util.MissingResourceException;
import java.util.Set;

/**
 * Base class for installation and uninstallation context.
 * The actual context passed to <a href="../screens/package-summary.html">screens</a>,
 * <a href="../actions/package-summary.html">actions</a> and
 * <a href="../formcomponents/package-summary.html">form components</a>
 * is either an {@link InstallerContext} or an {@link UninstallerContext}, depending on
 * whether they are part of the installer or uninstaller.
 *
 * @author ej-technologies GmbH
 * @see InstallerContext
 * @see UninstallerContext
 */
public interface Context {

    /**
     * Return the ISO code for the language that the installer is running with. If only the
     * principal language has been configured for the installer, this will always be the code for
     * the principal language. If additional languages have been configured, the user can
     * select a language or the language has been automatically determined by the installer, depending
     * on the settings on the "Languages" tab in the install4j GUI. If the installer chooses
     * a language automatically and none of the additional languages matches the system locale, the
     * principal language will be used.
     *
     * @return the ISO code for the language. This code is displayed in brackets after the language name in the
     *         install4j GUI.
     */
    String getLanguageId();

    /**
     * Get a message for a localization key.
     * This method is equivalent to
     * {@link #getMessage(String, Object[])} with the {@code arguments} parameter
     * set to {@code null}.
     *
     * @param key the localization key
     * @return the message
     * @throws MissingResourceException if the key cannot be found
     * @see #getMessage(String, Object[])
     */
    String getMessage(String key) throws MissingResourceException;

    /**
     * Get a message for a localization key.
     * The key can be either a system key as present in the {@code resource/messages/messages_*.utf8} files
     * or a key in a custom localization file.
     * <p>The search order is
     * <ul>
     * <li>custom localization file for actual language
     * <li>custom localization file for principal language
     * <li>system localization file provided by install4j
     * </ul>
     * <p>
     * The message is passed to
     * <pre>{@code
     * java.text.MessageFormat.format(message, arguments)
     * }</pre>
     * which replaces variables of the form {0} that are embedded in the message key.
     *
     * @param key the localization key
     * @param arguments the arguments as processed by the {@code java.text.MessageFormat} class. Can be {@code null}.
     * @return the message
     * @throws MissingResourceException if the key cannot be found
     */
    String getMessage(String key, Object... arguments) throws MissingResourceException;

    /**
     * Retrieves the installation directory selected by the user, or the default
     * installation directory if the user has not yet made a selection. Note that on
     * macOS this is usually {@code /Applications}. If you delete this directory,
     * it will have catastrophic consequences on macOS. To get the directory where files are
     * installed in a cross-platform way, please use {@link #getContentDirectory()} instead.
     * <p>
     * Due to update checking by the "Set installation directory" action or other calls of {@link InstallerContext#setInstallationDirectory},
     * the value returned by this method may not be the same as the one configured in the
     * install4j IDE. If you need to fall back to the default directory for some reason, you can use the return value
     * of {@link #getDefaultInstallationDirectory()} and call {@link InstallerContext#setInstallationDirectory(File)}.
     * </p>
     * @return the installation directory
     */
    File getInstallationDirectory();

    /**
     * Retrieves the default installation directory configured in the install4j IDE and resolved as suitable
     * for the current privilege level.
     * <p>
     * This method returns the same directory as {@link #getAllUsersDefaultInstallationDirectory()} if the {@code sys.installForAllUsers}
     * installer variable is set to {@code true}, otherwise it will return the same directory as
     * {@link #getUserWritableDefaultInstallationDirectory()}. The {@code sys.installForAllUsers} variable is {@code false} by default and
     * is changed by the "Set installation directory" action.
     * </p>
     * <p>
     * The return value of this method may change after a "Request privileges" action is executed successfully.
     * </p>
     * @see #getUserWritableDefaultInstallationDirectory()
     * @see #getAllUsersDefaultInstallationDirectory()
     * @return the default installation directory as configured in the install4j IDE either for all users or in a user-writable
     *          installation directory.
     */
    File getDefaultInstallationDirectory();

    /**
     * Retrieves the default installation directory for all users.
     * <p>
     *     The OS-specific parent directories are
     * </p>
     * <ul>
     *     <li>Windows: {@code %PROGRAMFILES%}</li>
     *     <li>macOS: {@code /Applications}</li>
     *     <li>Linux: {@code /opt}, fallback to {@code /usr/local} and then to {@code $HOME}</li>
     * </ul>
     * @see #getDefaultInstallationDirectory()
     */
    File getAllUsersDefaultInstallationDirectory();

    /**
     * Retrieves the default installation directory in a user-writable directory.
     * <p>
     *     The OS-specific parent directories are
     * </p>
     * <ul>
     *     <li>Windows: {@code %LOCALAPPDATA%\Programs}</li>
     *     <li>macOS: {@code $HOME/Applications}</li>
     *     <li>Linux: {@code $HOME}</li>
     * </ul>
     * @see #getDefaultInstallationDirectory()
     */
    File getUserWritableDefaultInstallationDirectory();

    /**
     * Retrieves the directory mapped to the "Installation directory" in the distribution tree.
     * On Windows, Linux, and Unix, this is the same as {@link #getInstallationDirectory()}.
     * For single bundle archives on macOS, this is {@code [Bundle name].app/Contents/Resources/app/}.
     * To reference an installed file in a cross-platform way, use this method and not {@link #getInstallationDirectory()}.
     *
     * @return the installation directory
     */
    File getContentDirectory();

    /**
     * Retrieves the directory where the resource files are present that have been configured on the
     * Installer-&gt;Custom Code &amp;Resources tab.
     * @return the resources directory
     */
    File getResourceDirectory();

    /**
     * Returns whether the installer or uninstaller is running in unattended mode.
     *
     * @return {@code true} or {@code false}.
     */
    boolean isUnattended();

    /**
     * Returns whether the installer or uninstaller is running in console mode.
     *
     * @return {@code true} or {@code false}.
     */
    boolean isConsole();

    /**
     * Returns whether the installer or uninstaller is running in GUI mode.
     *
     * @return {@code true} or {@code false}.
     */
    boolean isGui();

    /**
     * Returns whether the media set was an archive as opposed to an installer.
     *
     * @return {@code true} or {@code false}.
     */
    boolean isArchive();

    /**
     * Returns the application id of this project.
     *
     * @return the application id
     */
    String getApplicationId();

    /**
     * Returns the add-on application id if this project generates add-on installers.
     *
     * @return the add-on application id or the empty string if the project generates regular installers.
     */
    String getAddOnApplicationId();

    /**
     * Add a listener to receive installer events.
     *
     * @param listener the listener that should be added.
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     */
    void addInstallerEventListener(InstallerEventListener listener) throws NotSupportedInElevationException;

    /**
     * Remove a listener added with {@code addInstallerEventListener}.
     *
     * @param listener the listener that should be removed.
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     * @see #addInstallerEventListener(InstallerEventListener)
     */
    void removeInstallerEventListener(InstallerEventListener listener) throws NotSupportedInElevationException;

    /**
     * Programmatically switch to another screen in the installer without checking the condition of the target screen
     * or executing any actions.<p>
     * Corresponds to {@code gotoScreen(screen, false, false)}.
     * @param screen the screen to be activated.
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     * @see #gotoScreen(Screen, boolean, boolean)
     */
    void gotoScreen(Screen screen) throws NotSupportedInElevationException;

    /**
     * Programmatically switch to another screen in the installer. If
     * the given screen has not been registered, the operation will
     * fail. Registered screens can be obtained from the
     * {@code getScreenById(...)}, {@code getScreens(...)} and {@code getFirstScreen(...)} methods.
     * <p><b>Note:</b> This method only has a direct effect when called from a screen. When called from an action,
     * all remaining actions for the screen will be executed, and the most recent request for a screen change will then be performed.
     *
     * @param screen the screen to be activated.
     * @param checkCondition if the "Condition expression" of the target screen should be checked or not
     * @param executeActions if the associated actions of the current screen should be performed or not. This parameter
     *                       has no effect if this method is called from an action.
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     * @see #getScreens()
     * @see #getScreens(Class)
     * @see #getFirstScreen(Class)
     */
    void gotoScreen(Screen screen, boolean checkCondition, boolean executeActions) throws NotSupportedInElevationException;

    /**
     * Get a particular screen with the specified ID. You can display screen IDs in the install4j IDE.
     *
     * @param id the ID of the screen
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     * @return the screens, or {@code null} if no screen with the specified ID exists
     */
    Screen getScreenById(String id) throws NotSupportedInElevationException;

    /**
     * Get all registered screens.
     *
     * @return the screens
     */
    Screen[] getScreens();

    /**
     * Get all registered screens for a given screen class.
     *
     * @param screenClass the class of the screen.
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     * @return the screens or an empty array if no screen of this class could be found.
     */
    Screen[] getScreens(Class<?> screenClass) throws NotSupportedInElevationException;

    /**
     * Get the registered screen for a given screen class. If multiple screens of the same
     * class are registered, the first occurrence is returned. In that case,
     * {@link #getScreens(Class)} should be used.
     *
     * @param screenClass the class of the screen.
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     * @return the screen or {@code null} if no screen of this class could be found.
     * @see #getScreens(Class)
     */
    Screen getFirstScreen(Class<?> screenClass) throws NotSupportedInElevationException;

    /**
     * Get a particular action with the specified ID. You can display action IDs in the install4j IDE.
     *
     * @param id the ID of the action
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     * @return the action, or {@code null} if no action with the specified ID exists
     */
    Action getActionById(String id) throws NotSupportedInElevationException;

    /**
     * Get all registered actions for a specified screen.
     *
     * @param screen the screen whose associated actions should be returned
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     * @return the actions or an empty array if the specified screen has no associated actions.
     */
    Action[] getActions(Screen screen) throws NotSupportedInElevationException;

    /**
     * Get all registered actions for a specified screen and a given action class.
     *
     * @param actionClass the class of the action.
     * @param screen      the screen whose associated actions should be returned
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     * @return the actions or an empty array if no actions of this class could be found for the specified screen.
     */
    Action[] getActions(Class<? extends Action> actionClass, Screen screen) throws NotSupportedInElevationException;

    /**
     * Get the registered action for a specified screen and a given action class. If multiple actions of the same
     * class are registered, the first occurrence is returned. In that case,
     * {@link #getActions(Class, Screen)} should be used.
     *
     * @param actionClass the class of the action.
     * @param screen      the screen whose associated action should be returned or {@code null} if all screens should be searched
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     * @return the action or {@code null} if no action of this class could be found for the specified screen.
     * @see #getActions(Class, Screen)
     */
    Action getFirstAction(Class<? extends Action> actionClass, Screen screen) throws NotSupportedInElevationException;

    /**
     * Get the ID that the framework associates with an action.
     * <p>
     * This is the reverse method of {@link #getActionById(String)}.
     * @param action the action whose ID should be returned
     * @return the ID
     */
    String getId(Action action);

    /**
     * Get the ID that the framework associates with a screen.
     * <p>
     * This is the reverse method of {@link #getScreenById(String)}.
     * @param screen the screen whose ID should be returned
     * @return the ID
     */
    String getId(Screen screen);

    /**
     * Get the wizard context if the installer or uninstaller is running in GUI mode.
     * The wizard context allows you to modify the installer GUI.
     * The methods in the wizard context are not functional if the installer or uninstaller is running in unattended
     * or console mode, in those cases, they will do nothing.
     *
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     * @return the wizard context.
     */
    WizardContext getWizardContext();

    /**
     * Returns all additional files sets contained in this installer. If no file sets
     * have been defined in the install4j GUI, the returned collection will be empty.
     * The default file set is not contained in this list and will always be installed
     * by the "Install files" action.
     * The items in the collection will be in the same order as defined in the GUI.
     *
     * @return a collection containing {@code FileSetSetup} instances.
     * @see FileSetSetup
     */
    Collection<FileSetSetup> getFileSets();

    /**
     * Returns the file set for a specified ID. You can display
     * file set IDs in the install4j IDE.
     *
     * @param id the ID
     * @return the {@code FileSetSetup} or {@code null} if the ID doesn't exist.
     * @see FileSetSetup
     */
    FileSetSetup getFileSetById(String id);

    /**
     * Returns one file set with the specified name.
     *
     * @param name the name
     * @return the {@code FileSetSetup} or {@code null} if no file set with this name exists.
     * @see FileSetSetup
     */
    FileSetSetup getFileSetByName(String name);

    /**
     * Returns all installation components for this installer. If no installation components
     * have been defined in the install4j GUI, the returned collection will be empty.
     * The items in the collection will be in the same order as defined in the GUI.
     *
     * @return a collection containing {@code InstallationComponentSetup} instances.
     * @see InstallationComponentSetup
     */
    Collection<InstallationComponentSetup> getInstallationComponents();

    /**
     * Returns the installation component for a specified ID. You can display
     * component IDs in the install4j IDE.
     *
     * @param id the ID
     * @return the {@code InstallationComponentSetup} or {@code null} if the ID doesn't exist.
     * @see InstallationComponentSetup
     */
    InstallationComponentSetup getInstallationComponentById(String id);

    /**
     * Returns all launchers for this installer. If no launchers
     * have been defined in the install4j GUI, the returned collection will be empty.
     * The items in the collection will be in the same order as defined in the GUI.
     *
     * @return a collection containing {@code LauncherSetup} instances.
     * @see LauncherSetup
     */
    Collection<LauncherSetup> getLaunchers();

    /**
     * Returns the launcher for a specified ID. The ID of a launcher
     * can be displayed the install4j GUI by invoking Launcher-&gt;Show Launcher IDs from the main
     * menu when the Launchers section is active. The method is important for beans that
     * have a property of type {@code java.lang.String} whose property descriptor
     * has a context of type {@code CONTEXT_LAUNCHER_ID}. The property will contain a launcher
     * id that can be resolved at runtime with this method.
     *
     * @param id the ID
     * @return the {@code LauncherSetup} or {@code null} if the ID doesn't exist.
     * @see com.install4j.api.beaninfo.Install4JPropertyDescriptor#CONTEXT_LAUNCHER_ID
     */
    LauncherSetup getLauncherById(String id);

    /**
     * Returns whether the user has canceled the installation or not. Long-running actions
     * should check this method frequently to comply with a cancel request from the user.
     *
     * @return {@code true} or {@code false}.
     */
    boolean isCancelling();

    /**
     * Cancel the installer application without asking any questions. Rollback of already executed actions will be
     * performed.
     * <p>If a question has to be asked first, use {@link #runWithSuspendedActions(Runnable)} and call this method
     * from the Runnable.
     */
    void cancel();

    /** Execute some code while any running actions are suspended. Actions are only suspended when they check whether
     * the installer is being canceled, so the currently executing action may not be suspended immediately.
     * It makes sense to call this method as a reaction to an external event, such as a click on a button. For
     * example, you can then ask the user whether the installer application should be canceled and if yes, call
     * {@link #cancel()} in the Runnable.
     * @param runnable the code that will be executed while the actions are suspended
     */
    void runWithSuspendedActions(Runnable runnable);

    /**
     * If you encounter an exception in your code that cannot be handled, you can use this method to invoke
     * the default exception handling of the install4j runtime. An error report will be prepared, the user will be
     * notified, and the installer or uninstaller will quit. No rollback will be performed.
     * <p>This method is <b>for exceptional circumstances only</b>. Do not call this method if an action fails in an expected way:
     * if you return {@code false} from an action, its configurable failure strategy can be used to inform the user
     * of the failure and roll back previous actions.
     *
     * @param e the exception
     * @see com.install4j.api.Util#fatalError(Throwable)
     */
    void handleCriticalException(Throwable e);

    /**
     * Get the location where the installer will install a file in the distribution tree.
     * This method is important for actions with properties of type {@code java.io.File}.
     * The property values will be relative file names and contain encoded information
     * about installation roots. With this method, the installation location can be resolved.
     * Note that the result of this method can change over time as installer variables or the installation
     * directory are changed.
     *
     * @param archiveFile the relative file in the distribution tree
     * @return the location of the installed file
     */
    File getDestinationFile(File archiveFile);

    /**
     * Same as {@code getDestinationFile}, only with a path argument.
     *
     * @param archivePath the relative path in the distribution tree
     * @return the location of the installed file
     * @see #getDestinationFile(File)
     */
    File getDestinationFile(String archivePath);

    /**
     * Analyze a path in the distribution tree. Beans with {@code java.io.File} properties
     * can use this method to split paths into root information and relative information
     * and to resolve the installation location of a file in the distribution tree.
     * The {@code getDestinationFile(..)} methods directly give you the installed file location.
     *
     * @param archivePath the relative path in the distribution tree
     * @return the {@code FileInfo} information object
     * @see #getDestinationFile(File)
     * @see #getDestinationFile(String)
     */
    FileInfo getDestinationFileInfo(String archivePath);

    /**
     * Get an installer variable. In the install4j GUI, installer variables
     * are written as ${installer:variableName}. Please read the help topic on variables in the documentation
     * for more information.
     *
     * @param variableName the name of the installer variable.  Do not use the IDE syntax {@code ${installer:...}} for this parameter.
     * @return the variable value or {@code null} if the variable with the specified name was not defined.
     * @see #setVariable(String, Object)
     */
    Object getVariable(String variableName);

    /**
     * Convenience method to get an installer variable with a type of {@code java.lang.Boolean}.
     *
     * @param variableName the name of the installer variable. Do not use the IDE syntax {@code ${installer:...}} for this parameter.
     * @return {@code true} if the variable value equals {@code java.lang.Boolean.TRUE} or {@code false} otherwise
     *         (also if the variable has not been defined)
     * @see #getVariable(String)
     */
    boolean getBooleanVariable(String variableName);

    /**
     * Set an installer variable. Please read the help topic on variables in the documentation
     * for more information.
     *
     * @param variableName the name of the installer variable. Do not use the IDE syntax {@code ${installer:...}} for this parameter.
     * @param value        the new variable value
     * @see #getVariable(String)
     */
    void setVariable(String variableName, Object value);

    /**
     * Get the names of all defined installer variables. Please read the help topic on variables in the documentation.
     *
     * @return a set of variable names of type {@code java.lang.String}.
     * @see #getVariable(String)
     * @see #setVariable(String, Object)
     */
    Set<String> getVariableNames();

    /**
     * Gets the value of a compiler variable as it was defined for this media file
     * at compile time. Compiler variables are defined on the "Compiler variables" tab
     * and can be overridden for each media file. System compiler variables are also available
     * from this method. In the install4j IDE, compiler variables
     * are written as ${compiler:variableName}. Please read the help topic on variables in the documentation
     * for more information.
     *
     * @param variableName the name of the compiler variable. Do not use the IDE syntax {@code ${compiler:...}} for this parameter.
     * @return the variable value or {@code null} if the variable with the specified name was not defined.
     */
    String getCompilerVariable(String variableName);

    /**
     * Run a script. Beans that have properties of type {@code com.install4j.api.beans.ScriptProperty}
     * can run these user-defined scripts at runtime with this method. Script properties require
     * a property descriptor in the associated BeanInfo class that contains information about the return type
     * as well as any additional parameters.
     *
     * @param scriptProperty the value of the script property
     * @param bean           the bean that defines the script property
     * @param parameters     the additional parameters
     * @return the result of the script or {@code null} if the return type of the script is {@code void}
     * @throws Exception Scripts are user-defined and can throw all sorts of exceptions.
     * @see com.install4j.api.beaninfo.ScriptPropertyDescriptor
     */
    Object runScript(ScriptProperty scriptProperty, Bean bean, Object... parameters) throws Exception;

    /**
     * Get the current progress interface. If called by a screen or form component, this will be a default progress
     * interface where status messages and progress methods do nothing. A screen can replace that default progress
     * interface with a different implementation so that the status messages and progress methods invoked by the associated
     * actions are displayed on the screen. This is done by the "Installation" screen and the customizable
     * "Display progress" screen.
     * <p>During rollback, a special rollback dialog will be shown and make its progress interface available through
     * this method.
     *
     * @return the current progress interface
     * @see Screen#getProgressInterface(ProgressInterface)
     */
    ProgressInterface getProgressInterface();

    /**
     * Go forward in the screen sequence.
     * <p><b>Note:</b> This method only has a direct effect when called from a screen. When called from an action,
     * all remaining actions for the screen will be executed, and the most recent request for a screen change will then be performed.
     *
     * @param numberOfScreens the number of screens to move forward. If the resulting index is beyond the last screen, the
     *                        installer or uninstaller will quit.
     * @param checkCondition  if the "Condition expression" of the target screen should be checked or not
     * @param executeActions  if the associated actions of the current screen should be performed or not. This parameter
     *                        has no effect if this method is called from an action.
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     */
    void goForward(int numberOfScreens, boolean checkCondition, boolean executeActions) throws NotSupportedInElevationException;

    /**
     * Go backwards in the screen sequence. This is the sequence displayed on the "Screens" tab in the install4j GUI.
     * You can go backwards in the screen history with the {@code goBackInHistory} method.
     * <p><b>Note:</b> This method only has a direct effect when called from a screen. When called from an action,
     * all remaining actions for the screen will be executed, and the most recent request for a screen change will then be performed.
     *
     * @param numberOfScreens the number of screens to move backwards. If the resulting index is negative, a
     *                        {@code RuntimeException} will be thrown. You can pass {@code 0} to return to the current
     *                        screen from an action.
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     * @see #goBackInHistory(int)
     */
    void goBack(int numberOfScreens) throws NotSupportedInElevationException;

    /**
     * Go backwards in the screen history. This is not necessarily the same as the screen sequence, since the screen flow can be
     * changed programmatically.
     * <p><b>Note:</b> This method only has a direct effect when called from a screen. When called from an action,
     * all remaining actions for the screen will be executed, and the most recent request for a screen change will then be performed.
     *
     * @param numberOfScreens the number of screens to move backwards in the screen history. If the resulting index is negative, a
     *                        {@code RuntimeException} will be thrown. You can pass {@code 0} to return to the current
          *                        screen from an action.
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     * @see #goBack(int)
     */
    void goBackInHistory(int numberOfScreens) throws NotSupportedInElevationException;

    /**
     * Go backwards in the screen history to the target screen. If
     * the given screen has not been registered or was not displayed previously, the operation will
     * fail. Registered screens can be obtained from the
     * {@code getScreenById(...)}, {@code getScreens(...)} and {@code getFirstScreen(...)} methods.
     * <p><b>Note:</b> This method only has a direct effect when called from a screen. When called from an action,
     * all remaining actions for the screen will be executed, and the most recent request for a screen change will then be performed.
     *
     * @param targetScreen the screen to be activated.
     * @throws NotSupportedInElevationException if this method is called by an elevated action
     * @see #getScreens()
     * @see #getScreens(Class)
     * @see #getFirstScreen(Class)
     */
    void goBackInHistory(Screen targetScreen) throws NotSupportedInElevationException;

    /**
     * Get the location of an external file that was packaged with the installer. Beans that define properties of type {@code ExternalFile}
     * can access those files at runtime. When the installer is started, an external file
     * is located in a temporary directory. After the "Install files" action has run, the external file is
     * also copied to the {@code .install4j} directory in the installation directory.
     *
     * @param externalFile      the value of the {@code ExternalFile} property
     * @param installedLocation if {@code true}, this method returns the location in the {@code .install4j} directory,
     *                          otherwise the location in the temporary directory is returned. If called in an uninstaller, this argument has no effect and
     *                          the location in the {@code .install4j} directory will always be returned.
     * @return the resolved external file
     */
    File getExternalFile(ExternalFile externalFile, boolean installedLocation);

    /**
     * Get the location of a localized external file that was packaged with the installer. This
     * will return a file for the selected language as given by {@link #getLanguageId()}. If no
     * suitable localization exists, the file for the principal language is returned.
     * The remaining behavior of this method is identical to {@link #getExternalFile(ExternalFile, boolean)}.
     *
     * @param localizedExternalFile      the value of the {@code LocalizedExternalFile} property
     * @param installedLocation if {@code true}, this method returns the location in the {@code .install4j} directory,
     *                          otherwise the location in the temporary directory is returned. If called in an uninstaller, this argument has no effect and
     *                          the location in the {@code .install4j} directory will always be returned.
     * @return the resolved external file
     */
    File getExternalFile(LocalizedExternalFile localizedExternalFile, boolean installedLocation);

    /**
     * Determine if an error has occurred for any of the previous actions. After an action fails, this method
     * returns {@code true}. If the failure strategy of the failed action is configured in such a way that
     * the execution of the installer continues, you might want to react to the error in a subsequent action.
     * <p><b>Note:</b> This error flag is never cleared. You have to clear it manually with the {@code setErrorOccurred}
     * method. If you want to make sure that the error originates in a certain range of actions, you have to add
     * a "Run script" action just before that range and call {@code context.setErrorOccurred(false)} in order to
     * clear any earlier error.
     *
     * @return if an error has occurred or not
     * @see #setErrorOccurred(boolean)
     */
    boolean isErrorOccurred();

    /**
     * Clear or set the error flag that is described for {@code isErrorOccurred}.
     *
     * @param errorOccurred the new error flag
     * @see #isErrorOccurred()
     */
    void setErrorOccurred(boolean errorOccurred);

    /**
     * Register a variable that should be written to the response file when the installation is finished.
     * Variables to which form components are bound are automatically registered as response file variables.
     * Only strings, primitive wrappers, as well as string arrays, int arrays and {@code java.util.Date} values
     * can be encoded in the response file.
     * @param variableName the variable name
     */
    void registerResponseFileVariable(String variableName);

    /**
     * Reverses the action of {@link #registerResponseFileVariable(String)}. This also
     * works for system installer variables that are written to the response file
     * @param variableName the variable name
     */
    void unregisterResponseFileVariable(String variableName);

    /**
     * Register a comment for a variable that is written to the response file.
     * The comment can be a multi-line string. If non-empty lines do not start with a # character,
     * the # character is prepended. Compiler variables, installer variables, and i18n variables are
     * replaced in the comment string.
     *
     * @param variableName the variable name
     * @param comment the comment. Setting a comment to {@code null}, removes the comment.
     * @see #registerResponseFileVariable(String)
     */
    void registerResponseFileComment(String variableName, String comment);

    /**
     * Register a variable whose values should not be written to the installation log file.
     * You can use this method for variables which contain passwords or other information that
     * should not be saved to disk.
     *
     * @param variableName the variable name
     */
    void registerHiddenVariable(String variableName);


    /**
     * Triggers a reboot at the end of installation or uninstallation on Windows.
     *
     * @param askUser if true, the user will be asked if he wants to perform a reboot.
     */
    void triggerReboot(boolean askUser);

    /**
     * Returns whether a reboot is required on Windows.
     *
     * @return if a reboot is required or not
     */
    boolean isRebootRequired();

    /**
     * Quits the process in an ordered manner. All changes will be retained.
     *
     * @param exitCode the exit code.
     */
    void finish(int exitCode);

    /**
     * Returns if the code is running in the elevated helper process started by the "Request privileges" action.
     * In this case the code has full privileges, but it can interact with the GUI is a limited way, and some
     * methods in this context will throw a {@link NotSupportedInElevationException}.
     * @return {@code true} if elevated
     */
    boolean hasBeenElevated();

    /**
     * Execute a remoteCallable with full admin rights if possible.
     * <p>
     * If the current process has full admin rights, the remoteCallable is executed in-process.
     * Otherwise, if a 'require admin user' action and macOS or Windows Vista and higher has spawned a helper process
     * with full rights, it is executed by this process.
     *
     * <p>Note: Do not interact with the GUI or call methods like {@link com.install4j.api.Util#showMessage(String)}
     * in the code.
     *
     * @param remoteCallable the remoteCallable to execute
     * @param alwaysExecute also execute the remoteCallable if it cannot be executed with full admin rights.
     * @return the object returned by {@link RemoteCallable#execute}
     */
    Serializable runElevated(RemoteCallable remoteCallable, boolean alwaysExecute);

    /**
     * Execute a remoteCallable with the rights of the original user that started the installer.
     * <p>
     * If no elevation has been done by the 'require admin user' action, the remoteCallable is executed in-process.
     * Otherwise, it is executed by the original process.
     *
     * <p>Note: Do not interact with the GUI or call methods like {@link com.install4j.api.Util#showMessage(String)}
     * in the code.
     *
     * @param remoteCallable the remoteCallable to execute
     * @return the object returned by {@link RemoteCallable#execute}
     */
    Serializable runUnelevated(RemoteCallable remoteCallable);

    /** Initialize all lazily created screens. Screens provided by install4j are not created lazily, so this method
     * only has an effect, if you have custom screens that return {@code true} from
     * {@link Screen#isCreateLazily()}.
     */
    void initializeLazilyCreatedScreens();

    /** Add a JAR file or a classpath directory root to the classpath of the installer application.
     * This method if useful if you have lazily created custom screens in your project and you
     * have to add installed JAR files to the classpath before the {@link Screen#createComponent()} is called.
     * @param file the JAR file or the classpath directory root
     * @see Screen#isCreateLazily()
     */
    void addToClassPath(File file);

    /**
     * Returns an array with the command line arguments that were not recognized by install4j. The arguments are
     * in the same order as specified on the command line. The options recognized by install4j are removed.
     * <p>This is useful if you want to provide switches for console or unattended installers without resorting to
     * system properties (-Dname=value) or installer variables (-Vname=value).
     * @return the command line arguments
     */
    String[] getExtraCommandLineArguments();

    /**
     * Returns the ID of the media file that was used to create this installer.
     * @return the media file ID
     */
    String getMediaFileId();

    /**
     * Returns the version of the installer. For installer applications, this is the currently installed version.
     * @return the version
     */
    String getVersion();

    /**
     * Returns the name of the application.
     * This is the same as the {@code sys.fullName} installer variable on which all installer and i18n variables
     * have been replaced.
     * @return the application name
     */
    String getApplicationName();

}
