package com.install4j.api.beaninfo;

import java.beans.IntrospectionException;

/**
 * Property descriptor for properties that contain one or several file.
 * The supported property types are
 * <ul>
 * <li>java.io.File
 * <li>{@link com.install4j.api.beans.ExternalFile}
 * <li>{@link com.install4j.api.beans.LocalizedExternalFile}
 * <li>File[]
 * </ul>
 * <p>Using this class is not strictly required. In principle, you could also set values for the {@code ATTRIBUTE_*} constants in the property descriptor
 * of an unrelated property descriptor class.
 * @see com.install4j.api.beans.ExternalFile
 * @see com.install4j.api.beans.LocalizedExternalFile
 */
public class FilePropertyDescriptor extends Install4JPropertyDescriptor {

    /**
     * @see #create(String, Class, String, String, FileSelectionMode, String[], String)
     */
    public static final String ATTRIBUTE_SELECTION_MODE = "selectionMode";

    /**
     * @see #create(String, Class, String, String, FileSelectionMode, String[], String)
     */
    public static final String ATTRIBUTE_SUFFIXES = "suffixes";

    /**
     * @see #create(String, Class, String, String, FileSelectionMode, String[], String)
     */
    public static final String ATTRIBUTE_FILTER_NAME = "filterName";

    /**
     * @see #setFileContentType(FileContentType)
     */
    public static final String ATTRIBUTE_FILE_CONTENT_TYPE = "fileContentType";

    /**
     * Special context for properties of type {@code com.install4j.api.beans.ExternalFile} that tells the install4j GUI
     * to offer the user the possibility to select a file from the distribution tree instead. The selected file
     * will then be prepended with {@code ${installer:sys.installationDir}} and will not be packaged separately.
     * In that case, the enclosing action or screen can only be used if the "Install files" action has already run.
     */
    public static final String CONTEXT_EXTERNAL_OR_INTERNAL = "externalOrInternal";

    /**
     * Same as {@link #create(String, Class, String, String, FileSelectionMode, String[], String, String)}
     * with {@code suffixes}, {@code filterName} and {@code context} set to {@code null}.
     */
    public static FilePropertyDescriptor create(String propertyName, Class beanClass, String displayName, String shortDescription, FileSelectionMode selectionMode) {
        return create(propertyName, beanClass, displayName, shortDescription, selectionMode, null, null);
    }

    /**
     * Same as {@link #create(String, Class, String, String, FileSelectionMode, String[], String, String)}
     * with {@code suffixes} and {@code filterName} set to {@code null}.
     */
    public static FilePropertyDescriptor create(String propertyName, Class beanClass, String displayName, String shortDescription, FileSelectionMode selectionMode, String context) {
        return create(propertyName, beanClass, displayName, shortDescription, selectionMode, null, null, context);
    }

    /**
     * Same as {@link #create(String, Class, String, String, FileSelectionMode, String[], String, String)}
     * with {@code context} set to {@code null}.
     */
    public static FilePropertyDescriptor create(String propertyName, Class beanClass, String displayName, String shortDescription, FileSelectionMode selectionMode, String[] suffixes, String filterName) {
        return create(propertyName, beanClass, displayName, shortDescription, selectionMode, suffixes, filterName, null);
    }

    /**
     * Create a file property descriptor as with {@link Install4JPropertyDescriptor#create(String, Class, String, String)}
     * and additional configuration specific to file properties.
     * @param selectionMode the file selection mode when the user invokes the file chooser
     * @param suffixes the accepted suffixes in the file chooser
     * @param filterName the filter name in the file chooser
     * @param context the context. The context allows specifying different editors for the same property types.
     * See the {@code CONTEXT_*} properties for the built-in contexts.
     * @return the property descriptor
     */
    public static FilePropertyDescriptor create(String propertyName, Class beanClass, String displayName, String shortDescription, FileSelectionMode selectionMode, String[] suffixes, String filterName, String context) {
        try {
            FilePropertyDescriptor descriptor = createDescriptor(propertyName, beanClass);
            descriptor.setDisplayName(displayName);
            descriptor.setShortDescription(shortDescription);
            descriptor.setSelectionMode(selectionMode);
            descriptor.setSuffixes(suffixes);
            descriptor.setFilterName(filterName);
            descriptor.setContext(context);
            return descriptor;
        } catch (IntrospectionException e) {
            throw new RuntimeException(e);
        }
    }

    private void setSelectionMode(FileSelectionMode selectionMode) {
        if (selectionMode != null) {
            setValue(ATTRIBUTE_SELECTION_MODE, selectionMode);
        }
    }

    private void setSuffixes(String[] suffixes) {
        if (suffixes != null) {
            setValue(ATTRIBUTE_SUFFIXES, suffixes);
        }
    }

    private void setFilterName(String filterName) {
        if (filterName != null) {
            setValue(ATTRIBUTE_FILTER_NAME, filterName);
        }
    }

    private static FilePropertyDescriptor createDescriptor(String propertyName, Class beanClass) throws IntrospectionException {
        try {
            return new FilePropertyDescriptor(propertyName, beanClass);
        } catch (Exception ignored) {
        }
        return new FilePropertyDescriptor(propertyName, beanClass, "get" + capitalize(propertyName), null);
    }

    FilePropertyDescriptor(String propertyName, Class beanClass) throws IntrospectionException {
        super(propertyName, beanClass);
    }

    FilePropertyDescriptor(String propertyName, Class beanClass, String readMethod, String writeMethod) throws IntrospectionException {
        super(propertyName, beanClass, readMethod, writeMethod);
    }

    /**
     * Set the file content type. The install4j IDE can create some file types
     * as described in the documentation for {@link FileContentType}.
     * If you do not call this method, users will only be able to select the file, but there will be
     * no way to create or edit a selected file.
     * @param fileContentType the file content type
     * @return {@code this}, for chained calls on this property descriptor
     */
    public FilePropertyDescriptor setFileContentType(FileContentType fileContentType) {
        if (fileContentType != null) {
            setValue(ATTRIBUTE_FILE_CONTENT_TYPE, fileContentType);
        }
        return this;
    }

}
