package com.install4j.api.windows;

import com.install4j.runtime.installer.helper.Logger;
import com.install4j.runtime.installer.platform.win32.Win32UserInfo;

/**
 * Collection of static methods to create or delete users and local groups on Windows.
 * <p>If a helper process with elevated privileges has been created by the "Request privileges" action, the methods of
 * this class are pushed to the helper process. Please see the help topic on "Elevation Of Privileges" for more information.
 * @author ej-technologies GmbH
 */
public class WinUser {

    /**
     * Delete a user.
     * @param nameOrSid an account name or a SID in String form
     * @return {@code true} if successful or if the user doesn't exist
     */
    public static boolean deleteUser(String nameOrSid) {
        String accountName = Win32UserInfo.getAccountName(nameOrSid);
        return Win32UserInfo.deleteUser(accountName != null ? accountName : nameOrSid);
    }

    /**
     * Delete a local group.
     * @param nameOrSid an account name or a SID in String form
     * @return {@code true} if successful or if the local group doesn't exist
     */
    public static boolean deleteLocalGroup(String nameOrSid) {
        String accountName = Win32UserInfo.getAccountName(nameOrSid);
        return Win32UserInfo.deleteLocalGroup(accountName != null ? accountName : nameOrSid);
    }

    /**
     * Add a user and optionally a local group.
     * @param name the user name
     * @param password the password for the account
     * @param comment an optional comment. Can be {@code null}
     * @param groupCreationMode how the group should be created
     * @param localGroupNameOrSid a group name or optionally a SID if the group already exists. Can be {@code null} for {@link WinUser.GroupCreationMode#NONE}.
     * @param localGroupComment an optional comment for the local group. Can be {@code null}
     * @return the result of the operation
     */
    public static AddUserResult addUser(String name, String password, String comment, GroupCreationMode groupCreationMode, String localGroupNameOrSid, String localGroupComment) {
        if (name == null) throw new IllegalArgumentException("name required");
        if (password == null) throw new IllegalArgumentException("password required");
        if (groupCreationMode == null) throw new IllegalArgumentException("groupCreationMode required");

        if (localGroupNameOrSid != null && localGroupNameOrSid.trim().isEmpty()) {
            localGroupNameOrSid = null;
        }
        if (localGroupNameOrSid == null && groupCreationMode == GroupCreationMode.CREATE_IF_NECESSARY) {
            groupCreationMode = GroupCreationMode.NONE;
        }

        if (groupCreationMode != GroupCreationMode.NONE && localGroupNameOrSid == null) throw new IllegalArgumentException("localGroupName required");

        if (groupCreationMode == GroupCreationMode.NONE)  {
            localGroupNameOrSid = null;
        }

        if (localGroupNameOrSid != null) {
            String groupAccountName = Win32UserInfo.getAccountName(localGroupNameOrSid);
            if (groupAccountName != null) {
                localGroupNameOrSid = groupAccountName;
            }
        }
        Win32UserInfo.Result win32Result = Win32UserInfo.addUser(name, password, comment, localGroupNameOrSid, localGroupComment,
                groupCreationMode == GroupCreationMode.CREATE_IF_NECESSARY || groupCreationMode == GroupCreationMode.CREATE_OR_FAIL,
                groupCreationMode == GroupCreationMode.CREATE_OR_FAIL);
        if (win32Result == null) {
            return new AddUserResult(AddUserResultType.ERROR_OTHER);
        }
        switch (win32Result.getErrorCode()) {
            case Win32UserInfo.USER_CREATION_SUCCESS:
                return new AddUserResult(AddUserResultType.SUCCESS, win32Result.getUserSid(), win32Result.getGroupSid(), win32Result.isGroupCreated());
            case Win32UserInfo.NERR_GROUP_EXISTS:
            case Win32UserInfo.ERROR_ALIAS_EXISTS:
                return new AddUserResult(AddUserResultType.ERROR_GROUP_EXISTS);
            case Win32UserInfo.NERR_GROUP_NOT_FOUND:
                return new AddUserResult(AddUserResultType.ERROR_GROUP_NOT_FOUND);
            case Win32UserInfo.NERR_USER_EXISTS:
                return new AddUserResult(AddUserResultType.ERROR_USER_EXISTS);
            case Win32UserInfo.NERR_BAD_USER_NAME:
                return new AddUserResult(AddUserResultType.ERROR_BAD_USER_NAME);
            case Win32UserInfo.NERR_PASSWORD_TOO_SHORT:
                return new AddUserResult(AddUserResultType.ERROR_PASSWORD_REQUIREMENTS);
            case Win32UserInfo.ERROR_ACCESS_DENIED:
            case Win32UserInfo.NERR_NOT_PRIMARY:
                return new AddUserResult(AddUserResultType.ERROR_ACCESS_DENIED);
            default:
                Logger.getInstance().error(WinUser.class, "Could not create user. Win32 error code is " + win32Result.getErrorCode() + ", raw result: " + win32Result.getRawResult());
                return new AddUserResult(AddUserResultType.ERROR_OTHER);
        }
    }

    /**
     * The class of the objects returned by {@link WinUser#addUser}.
     */
    public static class AddUserResult {
        private AddUserResultType resultType;
        private String userSid;
        private String groupSid;
        private boolean groupCreated;

        private AddUserResult(AddUserResultType resultType, String userSid, String groupSid, boolean groupCreated) {
            this.resultType = resultType;
            this.userSid = userSid;
            this.groupSid = groupSid;
            this.groupCreated = groupCreated;
        }

        private AddUserResult(AddUserResultType resultType) {
            this.resultType = resultType;
        }

        /**
         * The result type (success / specific errors)
         * @return the result type
         */
        public AddUserResultType getType() {
            return resultType;
        }

        /**
         * The SID (Security Identifier) of the created user in String form. This can be used to unambiguously specify the created user
         * for actions such as 'Install a service' or 'Add Windows file rights'.
         * @return the user SID in String form
         */
        public String getUserSid() {
            return userSid;
        }

        /**
         * The SID (Security Identifier) of the used group in String form. This can be used to unambiguously specify the created group
         * for actions such as 'Add Windows file rights'.
         * @return the group sid in String form
         */
        public String getGroupSid() {
            return groupSid;
        }

        /**
         * Check if the local group was newly created
         * @return {@code true} if the group was created
         */
        public boolean isGroupCreated() {
            return groupCreated;
        }

        @Override
        public String toString() {
            return "AddUserResult{" +
                    "resultType=" + resultType +
                    ", userSid='" + userSid + '\'' +
                    ", groupSid='" + groupSid + '\'' +
                    ", groupCreated=" + groupCreated +
                    '}';
        }
    }

    /**
     *  The group creation mode.
     */
    public enum GroupCreationMode {
        /**
         * No group should be created and associated with the new user.
         */
        NONE("none"),

        /**
         * The new user should be added to an existing group.
         */
        USE_EXISTING("use existing"),

        /**
         * If the specified group does not exist, create it.
         */
        CREATE_IF_NECESSARY("create if necessary"),

        /**
         * Fail if the group already exists.
         */
        CREATE_OR_FAIL("create or fail");

        private final String name;

        GroupCreationMode(String name) {
            this.name = name;
        }

        @Override
        public String toString() {
            return name;
        }
    }

    /**
     * The type of result returned by {@link WinUser#addUser}
     */
    public enum AddUserResultType {

        /**
         * The user creation and the optional local group creation succeeded.
         */
        SUCCESS(""),

        /**
         * The current user has no access to the user database, or the
         * operation is only allowed on the primary domain controller of the domain.
         */
        ERROR_ACCESS_DENIED(""),

        /**
         * The group that should be newly created already exists.
         */
        ERROR_GROUP_EXISTS(""),

        /**
         * The group that should already exist could not be found.
         */
        ERROR_GROUP_NOT_FOUND(""),

        /**
         * The group that should already exist could not be found.
         */
        ERROR_USER_EXISTS(""),

        /**
         * The user name is invalid.
         */
        ERROR_BAD_USER_NAME(""),

        /**
         * The password does not meet the password policy requirements.
         * Check the minimum password length, password complexity, and password history requirements.
         */
        ERROR_PASSWORD_REQUIREMENTS(""),

        /**
         * Another type of error occurred during user creation.
         */
        ERROR_OTHER("");

        private final String verbose;

        AddUserResultType(String verbose) {
            this.verbose = verbose;
        }

        @Override
        public String toString() {
            return verbose;
        }
    }

    /**
     * Try to get an account name for a SID (Security Identifier).
     * @param sid the SID in String form
     * @return A qualified account name or {@code null} if the account could not be found.
     */
    public static String getAccountName(String sid) {
        return Win32UserInfo.getAccountName(sid);
    }

    /**
     * Try to get the user name (an unqualified account name) for a SID (Security Identifier).
     * @param sid the SID in String form
     * @return An unqualified account name or {@code null} if the account could not be found.
     */
    public static String getUserName(String sid) {
        return Win32UserInfo.getAccountName(sid, true);
    }

    /**
     * Get the user's SID (Security Identifier) of the current process.
     * @return A SID in String form or {@code null} if the method fails.
     */
    public static String getUserSid() {
        return Win32UserInfo.getUserSid();
    }
}
