package com.install4j.api.windows;

import com.install4j.runtime.installer.platform.win32.EnvVars;
import com.install4j.runtime.installer.platform.win32.Misc;

import java.io.IOException;
import java.util.Properties;

/** Collection of static methods to get and modify environment variables on Windows.
 * @author ej-technologies GmbH
 */
public class WinEnvVars {

    /** Sets a global environment variable. Needs Administrator rights.
     *
     * @param key the name of the environment variable
     * @param value the value of the environment variable
     * @throws IOException if the environment variable could not be set
     */
    public static void set(String key, String value) throws IOException {
        EnvVars.setWinNT(key, value, false);
    }

    /** Sets a user-specific environment variable.
     *
     * @param key the name of the environment variable
     * @param value the value of the environment variable
     * @throws IOException if the environment variable could not be set
     */
    public static void setUserSpecific(String key, String value) throws IOException {
        EnvVars.setWinNT(key, value, true);
    }

    /**
     * Returns a single environment variable.
     * @param key the name of the environment variable. Case does not matter.
     * @return the value of the environment variable or {@code null} if not defined
     */
    public static String getenv(String key) {
        return Misc.getenv(key);
    }

    /**
     * Returns all environment variables.
     * @return a property map with all environment variables and its values
     */
    public static Properties getenv() {
        return Misc.getenv();
    }


    /**
     * Return an environment variable as set in the Windows registry. The user-specific setting may be different
     * from the value returned in {@link #getenv()}. For example, the {@code PATH} environment variable is modified
     * by the install4j launcher and is not equal to the system setting. In addition, the value returned for
     * {@code PATH} will have no variable substitutions performed.
     * @param userSpecific If {@code true} the user-specific variant should be retrieved. Otherwise, the global setting
     *                     will be returned.
     * @return the value of the environment variable or {@code null} if not defined
     * @throws IOException if the environment variable could not be retrieved
     */
    public static String getFromRegistry(String key, boolean userSpecific) throws IOException {
        return EnvVars.getWinNT(key, userSpecific);
    }

    /**
     * Appends one or more directories to the PATH environment variable. Needs Administrator rights.
     * @param value the value to be appended to the path
     * @throws IOException if the environment variable could not be set
     */
    public static void appendToPath(String value) throws IOException {
        String previousPath = WinRegistry.getValue(RegistryRoot.HKEY_LOCAL_MACHINE, EnvVars.KEYNAME_GLOBAL, "PATH").toString();
        EnvVars.setWinNT("PATH", previousPath + ";" + value, false);
    }

    /**
     * Prepends one or more directories to the PATH environment variable. Needs Administrator rights.
     * @param value the value to be appended to the path
     * @throws IOException if the environment variable could not be set
     */
    public static void prependToPath(String value) throws IOException {
        String previousPath = WinRegistry.getValue(RegistryRoot.HKEY_LOCAL_MACHINE, EnvVars.KEYNAME_GLOBAL, "PATH").toString();
        EnvVars.setWinNT("PATH", value + ";" + previousPath, false);
    }

    /**
     * Appends one or more directories to the user-specific PATH environment variable.
     * @param value the value to be appended to the path
     * @throws IOException if the environment variable could not be set
     * @see #appendToPath(String)
     */
    public static void appendToUserPath(String value) throws IOException {
        String previousPath = WinRegistry.getValue(RegistryRoot.HKEY_CURRENT_USER, EnvVars.KEYNAME_USER, "PATH").toString();
        EnvVars.setWinNT("PATH", previousPath + ";" + value, true);
    }

    /**
     * Prepends one or more directories to the user-specific PATH environment variable.
     * @param value the value to be appended to the path
     * @throws IOException if the environment variable could not be set
     * @see #prependToPath(String)
     */
    public static void prependToUserPath(String value) throws IOException {
        Object userPath = WinRegistry.getValue(RegistryRoot.HKEY_CURRENT_USER, EnvVars.KEYNAME_USER, "PATH");
        String newUserPath;
        if (userPath == null) {
            newUserPath = value;
        } else {
            newUserPath =  value + ";" + userPath;
        }
        EnvVars.setWinNT("PATH", newUserPath, true);
    }
}
