package com.install4j.api.formcomponents;

import com.install4j.api.beans.Bean;
import com.install4j.api.context.Context;
import com.install4j.api.context.UserCanceledException;
import com.install4j.api.screens.Console;

import javax.swing.*;
import java.util.Map;

/**
 * All form components must implement this interface. You have to override all methods and provide a public constructor with no arguments.
 * It is recommended that you choose {@link AbstractFormComponent} as as a superclass.
 * <p>
 * Form components are configurable in the install4j GUI for screens that return {@code true} in their
 * {@code hasFormPanel} method. Their life-cycle is controlled by the framework.
 * </p>
 *
 * @see com.install4j.api.screens.FormPanelContainer
 * @author ej-technologies GmbH
 */
public interface FormComponent extends Bean {

    /**
     * This method is called by the framework to set the  {@code Context} just after the
     * form component has been constructed. This is either an {@code InstallerContext} or an
     * {@code UninstallerContext}, depending on whether the form screen is part of an
     * installer or uninstaller.
     * @param context the context.
     */
    void setContext(Context context);

    /**
     * At runtime, this method is called twice by the framework. Once, to set the  {@code FormEnvironment} just after the
     * screen has been constructed. The {@code getFormComponents()} method will return an empty array for that instance.
     * Secondly, after all form components have been instantiated, a new {@code FormEnvironment} is set whose
     * {@code getFormComponents()} method returns all form components in the form.
     * <p>At design time, this method is called repeatedly and the result of a call to  {@code getFormComponents()}
     * may be different over time.</p>
     *
     * @param formEnvironment the {@code FormEnvironment}
     * @see FormEnvironment#getFormComponents()
     */
    void setFormEnvironment(FormEnvironment formEnvironment);

    /**
     * Create the leading component to the left of the center component. This component will not grow beyond its preferred horizontal size.
     *
     * <p>In console or unattended mode, this method is never called.</p>
     * @return the leading component or {@code null} if no leading component should be created.
     */
    JComponent createLeftComponent();

    /**
     * Create the main component in the center. Depending on the return value of the {@code isFillCenterHorizontal}  method,
     * the component grows to fill all available horizontal space.
     * If you return {@code null}, the form component will not be added to the layout grid.
     *
     * <p>In console or unattended mode, this method is never called.</p>
     * @return the main component
     * @see #isFillCenterHorizontal()
     */
    JComponent createCenterComponent();

    /**
     * Determine whether the center component created by {@code createCenterComponent} should fill all available
     * horizontal space.
     * Note that you can configure the form not to grow horizontally in the install4j GUI, so the center component may be displayed with its
     * preferred horizontal size in any case.
     * @return {@code true} or {@code false}.
     * @see #createCenterComponent()
     */
    boolean isFillCenterHorizontal();

    /**
     * Determine how the center component created by {@code createCenterComponent} should should fill all available
     * vertical space.
     * If there are multiple form components on a form that return true for this method, the excess vertical space will
     * be distributed evenly.
     * Note that you can configure the form not to grow vertically in the install4j GUI, so the center component may be displayed with its
     * preferred height in any case.
     * @return {@code true} or {@code false}.
     * @see #createCenterComponent()
     */
    boolean isFillCenterVertical();

    /**
     * Create the trailing component to the right of the center component. This component will not grow beyond its preferred horizontal size.
     *
     * <p>In console or unattended mode, this method is never called.</p>
     * @return the trailing component or {@code null} if no trailing component should be created.
     */
    JComponent createRightComponent();

    /**
     * A form component can expose a well-known configuration object that is passed as a parameter to the "Initialization script"
     * property of every form component. Typically, you expose the main component contained in this form component,
     * such as a {@code JCheckBox} or a {@code JTextField}. If you return a non-null value in the method,
     * the {@code getConfigurationObject} method must return an object of that class.
     *
     * <p>In console or unattended mode, this method is never called.</p>
     *
     * @return the class of the configuration object or {@code null} if no configuration object should be exposed.
     * @see #getConfigurationObject()
     */
    Class<?> getConfigurationObjectClass();

    /**
     * If the {@code getConfigurationObjectClass} method
     * returns {@code null}, this method is not called, otherwise a non-null value of the type returned by
     * {@code getConfigurationObjectClass} has to be returned by this method.
     * <p>In console or unattended mode, this method is never called.</p>
     * @return the configuration object or {@code null} if {@code getConfigurationObjectClass} returns null as well.
     * @see #getConfigurationObjectClass()
     */
    Object getConfigurationObject();

    /**
     * This method is called by the framework when the user advances to the next screen in GUI or console mode.
     * If your form component expects user-input, this is the place to save it to an installer variable.
     * You can veto the screen change by returning {@code false} in this method.
     * @return whether to allow advancing to the next screen or not.
     * @see Context#setVariable(String, Object)
     */
    boolean checkCompleted();

    /**
     * This method is called by the framework when the initial state of the component should be evaluated.
     * This happens when the user first enters the screen that contains the form component. Whether repeated entries
     * to this screen will cause this method to be called depends on the value of the "Reset initialization on previous"
     * property that can be configured in the install4j GUI.
     * <p>
     * Initialization often involves replacing variables in user-configured property values.
     * </p>
     * @see com.install4j.api.beans.AbstractBean#replaceVariables(String)
     */
    void initialize();

    /**
     * This method is called by the framework just before the containing screen will be activated.
     * @see com.install4j.api.screens.Screen#willActivate()
     */
    void formWillActivate();

    /**
     * This method is called by the framework just after the containing screen has been activated.
     * @see com.install4j.api.screens.Screen#activated()
     */
    void formActivated();

    /**
     * This method is called by the framework just after the containing screen has been deactivated.
     * @see com.install4j.api.screens.Screen#deactivated()
     */
    void formDeactivated();

    /**
     * Handle the console mode. This method is called when the form component is processed in console mode.
     * You can use the {@code Console} object to interact with the user and replicate the GUI functionality
     * on the terminal.
     * <p>
     * Note: Screens with form panels have to call {@code FormEnvironment.handleConsole} in their
     * {@code handleConsole} method, otherwise this method will not be invoked. All pre-defined screens
     * in install4j comply with this requirement.
     * </p>
     * @param console the {@code Console} object
     * @return whether the installer or uninstaller can proceed with the next form component or whether the process should be canceled.
     * @throws UserCanceledException if the user cancels a question or notice. These exceptions are thrown by methods in the {@code Console} object.
     * @see com.install4j.api.screens.Screen#handleConsole(Console)
     * @see FormEnvironment#handleConsole(Console)
     */
    boolean handleConsole(Console console) throws UserCanceledException;

    /**
     * Handle the unattended mode. This method is called when the form component is processed in unattended mode.
     * There is no way to interact with the user. This method might be necessary to
     * mirror some behavior from the GUI mode, such as setting an installer variable.
     * @return whether the installer or uninstaller can proceed with the next form component or whether the process should be canceled.
     */
    boolean handleUnattended();

    /**
     * Sets whether the form component is enabled or not. This method is primarily intended for use in the
     * "Initialization script" property in the install4j GUI.
     * <p>It can also be used for the case where another form component wants to control the state of this form component.
     * For example, the "Check box" form component holds a list of coupled form components that are enabled and disabled with
     * calls to this method.
     * <p>{@code AbstractFormComponent} provides a default implementation that calls {@code setEnabled} on
     * the component returned by {@link #getConfigurationObject()}, saves the "enabled" flag and returns it in the
     * {@code isEnabled} method.
     * @param enabled whether to enable this form component or not
     * @see AbstractFormComponent
     * @see ComponentTuple
     * @see #isEnabled
     */
    void setEnabled(boolean enabled);

    /**
     * Returns whether the form component is enabled or not. This must reflect the value set with {@code setEnabled}.
     * <p>{@code AbstractFormComponent} provides a default implementation for both {@code isEnabled} and {@code setEnabled}.</p>
     * @return {@code true} or {@code false}.
     * @see #setEnabled(boolean)
     */
    boolean isEnabled();

    /**
     * Sets whether the form component is visible or not. This method is primarily intended for the 
     * "Initialization script" property in the install4j GUI.
     * <p>It can also be used for the case where another form component wants to control the state of this form component.
     * <p>{@code AbstractFormComponent} provides a default implementation that calls {@code setVisible} on
     * all components in the {@code ComponentTuple}, saves the "visible" flag and returns it in the
     * {@code isVisible} method.
     * @param visible whether this form component should be visible or not
     * @see AbstractFormComponent
     * @see ComponentTuple
     * @see #isVisible
     */
    void setVisible(boolean visible);

    /**
     * Returns whether the form component is visible or not. This must reflect the value set with {@code setVisible}.
     * <p>{@code AbstractFormComponent} provides a default implementation for both {@code isVisible} and {@code setVisible}.</p>
     * @return {@code true} or {@code false}.
     * @see #setVisible(boolean)
     */
    boolean isVisible();

    /**
     * This method is only called at design time when a user pastes form components or a screen with form components
     * from the clipboard. When form components are pasted, their IDs as returned by {@code FormEnvironment.getId}
     * change. Form components that keep a list of IDs for other form components should migrate those IDs in this method.
     * @param oldIdToNewId a map that maps the old IDs to the new IDs. An ID is of type {@code java.lang.String}.
     * @see FormEnvironment#getId(FormComponent)
     * @see FormEnvironment#getFormComponentById(String)
     */
    void migrateIds(Map<String, String> oldIdToNewId);

    /**
     * Transfer the focus to this form component.
     */
    void requestFocus();

    /**
     * This method is called in situations where the installer must disable all user inputs. If
     * your form component takes user input, such as for a checkbox or a text field, you should
     * return true.
     * @return if the component has user input
     */
    boolean hasUserInput();
}
