package com.install4j.api.events;

import com.install4j.api.context.Context;
import com.install4j.runtime.installer.InstallerVariables;
import com.install4j.runtime.installer.helper.InstallerUtil;

/**
 * The event object that is passed to installer event listeners for the variable change event type.
 * Event listeners are registered in the {@link com.install4j.api.context.InstallerContext}.
 * <p>This event object contains additional information about the changed variable.
 * The event type  and {@link EventType#VARIABLE_CHANGED}
 * produces this event object when fired.
 * @author ej-technologies GmbH
 * @see InstallerEventListener
 * @see com.install4j.api.context.InstallerContext#addInstallerEventListener(InstallerEventListener)
 *
 */
public class InstallerVariableEvent extends InstallerEvent {
    
    private String variableName;

    /**
     * The constructor is called by the framework.
     * @param source the event source
     * @param context the context
     * @param variableName the name of the changed variable
     */
    public InstallerVariableEvent(Object source, Context context, String variableName) {
        super(source, context, EventType.VARIABLE_CHANGED);
        this.variableName = variableName;
    }

    /**
     * Get the name of the variable that was changed.
     * @return the name of the variable
     */
    public String getVariableName() {
        return variableName;
    }

    @Override
    public String toString() {
        return super.toString() + " [variable " + variableName + "]";
    }

    @Override
    public String getVerbose() {
        if (InstallerVariables.getHiddenVariables().contains(variableName)) {
            return super.getVerbose() + ": " + variableName;
        } else {
            return super.getVerbose() + ": " + InstallerUtil.getVerboseVariableDefinition(variableName, getContext().getVariable(variableName));
        }
    }
}
