package com.install4j.api.context;

import com.install4j.runtime.installer.InstallerConstants;

import java.io.ObjectStreamException;

/**
 * Enumeration class that represents the different ways how install4j can handle the uninstallation of a file.
 * @author ej-technologies GmbH
 * @see FileOptions
 */
public enum UninstallMode {

    /**
     * Indicates that if the destination file was created by the installer, it will be deleted for uninstallation.
     * This is the default setting.
     */
    IF_CREATED("If created", InstallerConstants.UNINSTALL_IF_CREATED),

    /**
     * Indicates that the destination file won't be deleted for uninstallation.
     */
    NEVER("Never", InstallerConstants.UNINSTALL_NEVER),

    /**
     * Indicates that the uninstaller will always delete this file. Handle with care.
     */
    ALWAYS("Always", InstallerConstants.UNINSTALL_ALWAYS),

    /**
     * Indicates that the destination file won't be deleted when the uninstaller runs as part of an update.
     * Otherwise it behaves like {@code IF_CREATED}.
     */
    IF_CREATED_BUT_NOT_FOR_UPDATE("If created, but not for update", InstallerConstants.UNINSTALL_IF_CREATED_BUT_NOT_FOR_UPDATE),

    /**
     * Indicates that the destination file won't be deleted when the uninstaller runs as part of an update.
     * Otherwise it behaves like {@code ALWAYS}.
     */
    ALWAYS_BUT_NOT_FOR_UPDATE("Always, but not for update", InstallerConstants.UNINSTALL_NOT_FOR_UPDATE);

    /**
     * Convert int value to constant.
     * @param uninstall the int value
     * @return the constant
     */
    public static UninstallMode getFromIntValue(int uninstall) {
        for (UninstallMode uninstallMode : values()) {
            if (uninstallMode.intValue == uninstall) {
                return uninstallMode;
            }
        }
        return null;
    }

    private final String description;
    private final int intValue;

    UninstallMode(String description, int value) {
        this.description = description;
        this.intValue = value;
    }

    @Override
    public String toString() {
        return description;
    }

    /**
     * Convert to an int value.
     * @return the int value
     */
    public int getIntValue() {
        return intValue;
    }

    private Object readResolve() throws ObjectStreamException {
        return getFromIntValue(intValue);
    }

}
