package com.install4j.api.beaninfo;

/**
 * A property converter translates between a string representation and the actual object state and provides
 * an optional {@link PropertyCustomizer property customizer}.
 * Property converters are registered on a per-bean basis with
 * {@link Install4JBeanInfo#setPropertyConverters(PropertyConverter[])}.
 * <p>This class and the {@link PropertyConverter} class are similar to the {@link java.beans.PropertyEditor} class
 * which was not used in the install4j API since the editing concepts of the install4j IDE do not quite match with the
 * ideas in the java.beans API.
 * <p>In concept, this class is similar to the {@link EnumerationMapper} class.
 */
public interface PropertyConverter {

    /**
     * A default context that you can use in the {@link #getContext()}  method if there is only one
     * property converter for the property type returned by {@link #getPropertyClass()}
     */
    String DEFAULT_CONTEXT = "defaultConverterContext";

    /**
     * Returns the property type for which this property converter is eligible. Derived classes are also captured.
     * @return the property type
     */
    Class getPropertyClass();

    /**
     * The property descriptor of a property that wants to use this property converter must have a non-null context
     * that corresponds to the return value of this method. In this way, the runtime knows that this property converter
     * should be used for displaying and editing a property. The context must not be globally unique,
     * property converters of other property types can use the same context value.
     * <p>For classes that do not have default handlers in install4j,
     * the property converter will also be used if this method returns {@link #DEFAULT_CONTEXT}, the 
     * {@link #getPropertyClass()} method returns the class of the property and the context of the property is not set.
     * However, it is safer not to rely on that behavior.
     *
     * @return the context that will be used by the properties
     */
    String getContext();

    /**
     * Converts the property value to a string value. The string value will be displayed in the install4j IDE.
     * If {@link #supportsFromString()} returns {@code true}, the string value must be valid input for
     * {@link #fromString(String)} and return an object with the same internal state.
     * @param object the property value of the type returned by {@link #getPropertyClass()} or possibly of a derived class
     * @return the string value
     */
    String toString(Object object);

    /**
     * Returns if the property converter is able to convert a string representation to an object.
     * If yes, the user will be able to edit the string returned by {@link #toString(Object)} in the install4j IDE.
     * @return {@code true} if a string can be converted to an object
     */
    boolean supportsFromString();

    /**
     * Converts a string representation to an object of type {@link #getPropertyClass()}.
     * When editing of the property is completed in the install4j IDE, this method will be called to construct an object that will be set as the
     * property value. This method is only called if {@link #supportsFromString()} returns {@code true}.
     * @param string the string representation
     * @return the object
     */
    Object fromString(String string);


    /**
     * Returns the class of the optional property customizer. If you want to offer a dialog where the user can edit the
     * property value, you have to implement {@link PropertyCustomizer} and return the name of your property customizer class,
     * otherwise return {@code null}.
     * <p>If a non-null value is returned, the user will see a "..." button in the install4j IDE that brings up the dialog
     * with the property customizer.
     * @return the property customizer class or {@code null}
     */
    Class getPropertyCustomizerClass();
}
