package com.install4j.api;

import com.install4j.api.laf.LookAndFeelEnhancer;
import com.install4j.runtime.installer.frontend.GUIHelper;
import com.install4j.runtime.installer.frontend.LookAndFeelHelper;
import org.jetbrains.annotations.NotNull;

import java.awt.*;

/**
 * This class contains methods that deal with global properties of the installer application GUI.
 * <p>
 * When running in console or unattended mode, these methods can still be called, although they may not have
 * any effect or return any helpful information.
 * </p>
 *
 * @author ej-technologies GmbH
 */
public class UiUtil {
    /**
     * Returns if the desktop environment is in dark mode or not.
     * <p>
     * To check whether the installer application is running with a dark look and feel, you should call
     * {@link #isDarkUi()}
     * instead. If you have selected the FlatLaf look and feel for installer applications with dark mode
     * auto-detection enabled, both methods will return the same value.
     */
    public static boolean isDarkDesktop() {
        return LookAndFeelHelper.isDarkDesktop();
    }

    /**
     * Returns if the installer application is running in dark mode or not.
     * <p>
     * This is a convenience method for calling
     * <pre>{@code
     *  LookAndFeelEnhancer.getInstance().isDark()
     *}</pre>
     * @see LookAndFeelEnhancer#isDark()
     */
    public static boolean isDarkUi() {
        return getLookAndFeelEnhancer().isDark();
    }

    /**
     * Switch between dark and light mode and update all windows with the new mode.
     * <p>
     * This will only have an effect if {@link LookAndFeelEnhancer#isDarkModeSwitchingSupported()}
     * returns {@code true} and if the installer application is running in GUI mode.
     * </p>
     * @param dark {@code true} for dark mode
     */
    public static void setDarkUi(boolean dark) {
        LookAndFeelHelper.setDarkUi(dark);
    }

    /**
     * Returns the currently active look and feel info.
     */
    @NotNull
    public static LookAndFeelEnhancer getLookAndFeelEnhancer() {
        return LookAndFeelHelper.getLookAndFeelHandler();
    }

    /**
     * Get the topmost window in the current JVM.
     *
     * @return the window or {@code null} if the installer application is running in console or unattended mode.
     */
    public static Window getParentWindow() {
        return GUIHelper.getParentWindow();
    }

    /**
     * Get the current install4j wizard window.
     *
     * @return the window or {@code null} if the installer application is running in console or unattended mode.
     */
    public static Window getWizardWindow() {
        return GUIHelper.getCurrentWizardWindow();
    }
}
