package com.install4j.api.windows.service;

import com.install4j.runtime.installer.platform.win32.Win32Services;

import java.io.Serializable;

/**
 * The service configuration passed to {@link WinServices}. If you don't set a property, the original value will be preserved for existing services.
 * @author ej-technologies GmbH
 */
public class ServiceConfiguration implements Serializable {
    private String binaryName;
    private String displayName;
    private ServiceStartType startType;
    private String dependencies;
    private String description;
    private Boolean delayedAutoStart;
    private Boolean restartOnFailure;
    private int restartMillis = Win32Services.DEFAULT_RESTART_MILLIS;
    private int maxRestarts = Win32Services.DEFAULT_MAX_RESTARTS;
    private int resetSeconds = Win32Services.DEFAULT_RESET_SECONDS;

    private ServiceAccount serviceAccount;
    private String otherAccountName;
    private String otherAccountPassword;

    /**
     * The path to the binary. This can be quoted and contain arguments
     * @param binaryName the binary name
     * @return this object
     */
    public ServiceConfiguration binaryName(String binaryName) {
        this.binaryName = binaryName;
        return this;
    }

    /**
     * The display name. This is different from the service name which serves as identifier.
     * @param displayName the display name
     * @return this object
     */
    public ServiceConfiguration displayName(String displayName) {
        this.displayName = displayName;
        return this;
    }

    /**
     * The start type
     * @param startType the start type
     * @return this object
     */
    public ServiceConfiguration startType(ServiceStartType startType) {
        this.startType = startType;
        return this;
    }

    /**
     * A comma-separated list of dependencies
     * @param dependencies the dependencies
     * @return this object
     */
    public ServiceConfiguration dependencies(String dependencies) {
        this.dependencies = dependencies;
        return this;
    }

    /**
     * An additional description for the service
     * @param description the description
     * @return this object
     */
    public ServiceConfiguration description(String description) {
        this.description = description;
        return this;
    }

    /**
     * If startType is set to {@code AUTO}, you can specify with this property that the service is started with a short delay after all other auto-start services.
     * @param delayedAutoStart {@code true} if the service start should be delayed
     * @return this object
     */
    public ServiceConfiguration delayedAutoStart(Boolean delayedAutoStart) {
        this.delayedAutoStart = delayedAutoStart;
        return this;
    }

    /**
     * If set to true, the service will be restarted after a period defined with {@link #restartMillis} (defaults to one second) if it was terminated with an error
     * @param restartOnFailure {@code true} if a restart should be performed
     * @return this object
     */
    public ServiceConfiguration restartOnFailure(Boolean restartOnFailure) {
        this.restartOnFailure = restartOnFailure;
        return this;
    }

    /**
     * The restart period when {@link #restartOnFailure} is set to true. Defaults to one second.
     * @param restartMillis the restart period in milliseconds
     * @return this object
     */
    public ServiceConfiguration restartMillis(int restartMillis) {
        this.restartMillis = restartMillis;
        return this;
    }

    /**
     * The maximum number of restarts when {@link #restartOnFailure} is set to true. Defaults to 0, which means unlimited restarts. With {@link #resetSeconds} you can define the period after which this counter will be reset.
     * This should be a value between 0 and 2 to fit into the Windows service UI.
     * @param maxRestarts the number of restarts to try
     * @return this object
     */
    public ServiceConfiguration maxRestarts(int maxRestarts) {
        this.maxRestarts = maxRestarts;
        return this;
    }

    /**
     * The time without failure in seconds after which to reset the restart counter configured with {@link #maxRestarts}. Defaults to 0.
     * @param resetSeconds the time in seconds
     * @return this object
     */
    public ServiceConfiguration resetSeconds(int resetSeconds) {
        this.resetSeconds = resetSeconds;
        return this;
    }

    /**
     * The service account to be used. Can be one of the predefined Windows accounts or {@code OTHER}. If it is {@code OTHER}, you have to specify
     * otherAccountName and otherAccountPassword
     * @param serviceAccount the account to be used
     * @return this object
     */
    public ServiceConfiguration serviceAccount(ServiceAccount serviceAccount) {
        this.serviceAccount = serviceAccount;
        return this;
    }

    /**
     * The service account name to be used. The serviceAccount property will be set to {@code OTHER} when you call this method.
     * The name must be in the form DomainName\UserName. If the account belongs to the built-in domain, you can specify .\UserName.
     * @param otherAccountName the account name or SID
     * @return this object
     */
    public ServiceConfiguration otherAccountName(String otherAccountName) {
        serviceAccount = ServiceAccount.OTHER;
        this.otherAccountName = otherAccountName;
        return this;
    }

    /**
     * The service account password to be used. The serviceAccount property will be set to {@code OTHER} when you call this method.
     * @param otherAccountPassword the account password
     * @return this object
     */
    public ServiceConfiguration otherAccountPassword(String otherAccountPassword) {
        serviceAccount = ServiceAccount.OTHER;
        this.otherAccountPassword = otherAccountPassword;
        return this;
    }

    String getBinaryName() {
        return binaryName;
    }

    String getDisplayName() {
        return displayName;
    }

    ServiceStartType getStartType() {
        return startType;
    }

    String getDependencies() {
        return dependencies;
    }

    ServiceAccount getServiceAccount() {
        return serviceAccount;
    }

    String getOtherAccountName() {
        return otherAccountName;
    }

    String getOtherAccountPassword() {
        return otherAccountPassword;
    }

    String getDescription() {
        return description;
    }

    Boolean getDelayedAutoStart() {
        return delayedAutoStart;
    }

    Boolean getRestartOnFailure() {
        return restartOnFailure;
    }

    public Integer getRestartMillis() {
        return restartMillis;
    }

    public Integer getMaxRestarts() {
        return maxRestarts;
    }

    public Integer getResetSeconds() {
        return resetSeconds;
    }
}
