package com.install4j.api.macos;

import com.install4j.runtime.installer.platform.macos.ServiceManagement;

/**
 * Methods for controlling helper executables that live inside an app’s main bundle on macOS.
 * <p>This API is supported starting with macOS 13.
 * Launch agents and daemons are only supported by single bundle archives and must be configured in the media wizard.
 * The methods handling the main bundle can be used for all media types.</p>
 * <p>
 * General info about these mechanisms can be found in the documentation of the
 * <a href="https://developer.apple.com/documentation/servicemanagement/smappservice">Apple Service Management Framework</a>
 * </p>
 * @author ej-technologies GmbH
 */
public class MacServiceManagement {

    /**
     * Registers the main bundle as a login item.
     *
     * @return the result of the operation, represented as a {@link ServiceManagementResult}
     */
    public static ServiceManagementResult registerMainBundleAsLoginItem()  {
        return ServiceManagement.registerMain();
    }

    /**
     * Unregisters the main bundle as a login item.
     *
     * @return the result of the operation, represented as a {@link ServiceManagementResult}
     */
    public static ServiceManagementResult unregisterMainBundleAsLoginItem() {
        return ServiceManagement.unregisterMain();
    }

    /**
     * Retrieves the status of the main bundle login item registration.

     * @return the status of the registration, represented as an {@link AppServiceStatus} enum value or {@code null} if the status could not be determined.
     */
    public static AppServiceStatus getMainBundleLoginItemStatus() {
        return ServiceManagement.getMainBundleStatus();
    }

    /**
     * Registers a launch agent specified by its property list (plist) file name. The launch agent must be configured
     * in the single bundle media wizard. The plist file is located inside the main bundle in {@code Contents/Library/LaunchAgents}.
     * <p>The launch agent is immediately bootstrapped and may begin running. In addition, launch agents registered with this
     * method bootstrap on each subsequent login.</p>
     *
     * @param plistName the name of the property list (.plist) file that defines the launch agent
     *                  configuration.
     *
     * @return the result of the operation, represented as a {@link ServiceManagementResult}
     */
    public static ServiceManagementResult registerLaunchAgent(String plistName) {
        return ServiceManagement.register(checkPlistName(plistName), false);
    }

    /**
     * Unregisters a launch agent specified by its property list (plist) file name. The launch agent must be configured
     * in the single bundle media wizard. The plist file is located inside the main bundle in {@code Contents/Library/LaunchAgents}.
     * <p>If the launch agent is currently running, it will be terminated.</p>
     *
     * @param plistName the name of the property list (.plist) file that defines the launch agent
     *                  configuration.
     *
     * @return the result of the operation, represented as a {@link ServiceManagementResult}
     */
    public static ServiceManagementResult unregisterLaunchAgent(String plistName) {
        return ServiceManagement.unregister(checkPlistName(plistName), false);
    }

    /**
     * Retrieves the status of a launch agent specified by its property list (plist) file name.
     * The plist file is expected to be located within the main bundle at {@code Contents/Library/LaunchAgents}.
     *
     * @param plistName the name of the property list (.plist) file that defines the launch agent configuration.
     * @return the status of the launch agent, represented as an {@link AppServiceStatus} enum value or {@code null} if the status could not be determined.
     */
    public static AppServiceStatus getLaunchAgentStatus(String plistName) {
        return ServiceManagement.getStatus(checkPlistName(plistName), false);
    }

    /**
     * Registers a launch daemon specified by its property list (plist) file name. The launch daemon must be configured
     * in the single bundle media wizard. The plist file is located inside the main bundle in {@code Contents/Library/LaunchDaemons}.
     * <p>The system won’t bootstrap the launch daemon until an admin approves the launch daemon in System Preferences.
     * The system bootstraps launch daemon registered with this method and approved by an admin on each subsequent boot.</p>
     *
     * @param plistName the name of the property list (.plist) file that defines the launch daemon
     *                  configuration.
     *
     * @return the result of the operation, represented as a {@link ServiceManagementResult}
     */
    public static ServiceManagementResult registerLaunchDaemon(String plistName) {
        return ServiceManagement.register(checkPlistName(plistName), true);
    }

    /**
     * Unregisters a launch daemon specified by its property list (plist) file name. The launch daemon must be configured
     * in the single bundle media wizard. The plist file is located inside the main bundle in {@code Contents/Library/LaunchDaemons}.
     * <p>If the launch daemon is currently running, it will be terminated.</p>
     *
     * @param plistName the name of the property list (.plist) file that defines the launch daemon
     *                  configuration.
     *
     * @return the result of the operation, represented as a {@link ServiceManagementResult}
     */
    public static ServiceManagementResult unregisterLaunchDaemon(String plistName) {
        return ServiceManagement.unregister(checkPlistName(plistName), true);
    }

    /**
     * Retrieves the status of a launch daemon specified by its property list (plist) file name.
     * The plist file is expected to be located within the main bundle at {@code Contents/Library/LaunchDaemons}.
     *
     * @param plistName the name of the property list (.plist) file that defines the launch daemon configuration.
     * @return the status of the launch daemon, represented as an {@link AppServiceStatus} enum value or {@code null} if the status could not be determined.
     */
    public static AppServiceStatus getLaunchDaemonStatus(String plistName) {
        return ServiceManagement.getStatus(checkPlistName(plistName), true);
    }

    /**
     * Opens the system settings in the "Login Items" section, allowing the user to view or manage applications
     * that are set to launch at login.
     *
     * @return if opening the system settings is supported.
     */
    public static boolean openSystemSettingsLoginItems() {
        return ServiceManagement.openSystemSettingsLoginItems();
    }

    /**
     * Result of a service management operation.
     */
    public enum ServiceManagementResult {
        OK,
        ERROR_UNSUPPORTED,
        ERROR_UNKNOWN,
        ERROR_REQUIRES_APPROVAL,
        ERROR_INTERNAL_FAILURE,
        ERROR_INVALID_SIGNATURE,
        ERROR_AUTHORIZATION_FAILURE,
        ERROR_TOOL_NOT_VALID,
        ERROR_JOB_NOT_FOUND,
        ERROR_SERVICE_UNAVAILABLE,
        ERROR_JOB_PLIST_NOT_FOUND,
        ERROR_JOB_MUST_BE_ENABLED,
        ERROR_INVALID_PLIST,
        ERROR_LAUNCH_DENIED_BY_USER,
        ERROR_ALREADY_REGISTERED
    }

    /**
     * Enumeration representing the possible statuses of an application service or process.
     * This enum can be used to determine the current state of services such as launch agents
     * or launch daemons managed through the macOS service management framework.
     */
    public enum AppServiceStatus {
        NOT_REGISTERED,
        ENABLED,
        REQUIRES_APPROVAL,
        NOT_FOUND
    }

    private static String checkPlistName(String plistName) {
        if (plistName == null) {
            throw new IllegalArgumentException("plistName must not be null");
        } else if (!plistName.endsWith(".plist")) {
            return plistName + ".plist";
        } else {
            return plistName;
        }
    }
}
