package com.install4j.api.launcher;

import com.exe4j.Controller;
import com.install4j.runtime.installer.helper.InstallerUtil;
import com.install4j.runtime.installer.helper.versionspecific.VersionSpecificHelper;
import com.install4j.runtime.launcher.MacLauncher;
import com.install4j.runtime.launcher.util.SingleInstance;

import java.util.EventListener;

/**
 *
 * <p>This class allows you to register a listener to receive startup
 * events in single instance mode on Microsoft Windows and
 * file open events on macOS.</p>
 *
 * On macOS, this is a facade for the platform-specific Java extensions.
 * If you want to use additional features, please use the com.apple.eawt
 * classes directly that are available in all pre-Java 9 macOS JREs or
 * the new Java 9 java.awt.desktop functionality.
 *
 * @author ej-technologies GmbH
 *
 **/
public class StartupNotification {
    private static Listener listener;

    /**
     * Register a listener to receive startup events in single instance mode on
     * Microsoft Windows or to receive file open events on macOS.
     * Requests that are already queued may be invoked immediately on the startup listener before this call returns.
     * Subsequent calls will be from different threads.
     * <p>
     *  <b>On macOS, you must register this listener on the main thread of your application</b>.
     *  Otherwise the initial file open event may be lost.
     *  If the application bundle is started through a file association, the file name will also be passed to the
     *  listener and not as an argument to the main class. If your application can only handle the file open event
     *  once it is initialized, you need to store the event and handle it later.
     *  In particular, if you are using JavaFX on macOS, and the application is started through a file association, the JavaFX platform
     *  will not be initialized at that time and {@code Platform.runLater} will throw an {@link IllegalStateException}.
     * </p>

     * @param listener the listener
     */
    public static void registerStartupListener(final Listener listener) {
        if (InstallerUtil.isWindows()) {
            StartupNotification.listener = listener;
            if (listener != null) {
                Controller.registerStartupListener(new ListenerAdapter());
            }
        } else if (InstallerUtil.isMacOS() && !MacLauncher.isCommandLine()) {
            VersionSpecificHelper.setMacStartupListener(listener);
        } else {
            SingleInstance.registerStartupListener(listener::startupPerformed);
        }
    }


    /**
     * Sets if the Quit command should be handled on macOS (that means if the application
     * should quit when the command is invoked). This is true be default.
     * @param handleQuit true if quit should be handled
     */
    public static void setHandleQuit(boolean handleQuit) {
        if (InstallerUtil.isMacOS()) {
            VersionSpecificHelper.setMacHandleQuit(handleQuit);
        }
    }


    /**
     * The interface for startup events in single instance mode or for open file events on macOS.
     *
     * @author ej-technologies GmbH
     *
    */
    public interface Listener extends EventListener {
        /**
         * Will be called when the executable is started again or when a file open event is received.
         * Note that each invocation may be from a separate thread. If you have to serialize
         * calls into this method, please synchronize the implementation.
         * @param parameters the parameters of the invocation, or the file to open on macOS.
         * For multiple files, files are surrounded by double-quotes and separated by spaces.
         *  <p><b>Note:</b> any default arguments defined for the launcher will not be
         *  passed again here</p>.
         */
        void startupPerformed(String parameters);
    }

    private static class ListenerAdapter implements Controller.StartupListener {
        @Override
        public void startupPerformed(String parameters) {
            if (listener != null) {
                listener.startupPerformed(parameters);
            }
        }
    }
}
