package com.install4j.api.beans;

import com.install4j.runtime.installer.InstallerConstants;
import com.install4j.runtime.util.IconHelper;

import java.io.File;

/**
 * Property type for files from outside the distribution tree to be used in beans.
 * Properties of type {@code java.io.File} refer to files in the distribution tree. For these properties, in the install4j GUI
 * a distribution file chooser will be shown that offers only files configured in the "Files" section
 * of the install4j GUI.
 * <p>If you use {@code ExternalFile} as the property type, the install4j GUI will show a regular file chooser
 * for the property that lets you select files from arbitrary locations. The selected file will be packaged with the installer
 * during the compilation phase. The name of the file is changed to a unique name during the compilation.
 * It will be available in a temporary directory when the installer or uninstaller is started and extracted
 * to the {@code .install4j} directory in the installation directory by the "Install files" action.
 *
 * <p><b>Note</b>:At runtime, you have to call {@code Context.getExternalFile} to transform the property to
 * an absolute file.
 *
 * @see com.install4j.api.context.Context#getExternalFile(ExternalFile, boolean)
 * @author ej-technologies GmbH
 */
public class ExternalFile extends File {

    private final String originalName;

    /**
     *  The constructor will be invoked by the framework.
     */
    public ExternalFile(String pathname) {
        this(pathname, "");
    }

    /**
     *  The constructor will be invoked by the framework.
     */
    public ExternalFile(String pathname, String originalName) {
        super(pathname);
        this.originalName = originalName;
    }

    /**
     * At runtime, the file name will be generated to avoid naming conflicts. This method returns the original
     * file name.
     * @return the file name
     */
    public String getOriginalName() {
        return originalName;
    }

    // This overridden method is required, otherwise the XML encoder will not work correctly
    @Override
    public boolean equals(Object obj) {
        return super.equals(obj);
    }

    @Override
    public boolean exists() {
        String path = getPath();
        if (path.startsWith(InstallerConstants.ICON_RESOURCE_PREFIX)) {
            return IconHelper.getIconUrl(path) != null;
        } else {
            return super.exists();
        }
    }
}
