package com.install4j.api.beaninfo;

import com.install4j.api.beans.Bean;
import com.install4j.api.beans.LocalizedExternalFile;
import com.install4j.api.beans.ScriptProperty;

import javax.swing.*;
import java.beans.BeanDescriptor;
import java.beans.PersistenceDelegate;
import java.beans.PropertyDescriptor;
import java.beans.SimpleBeanInfo;
import java.lang.reflect.Array;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * Base class for bean infos. It is recommended to derive your bean info from one of the derived classes
 * which are specific to <a href="../screens/package-summary.html">screens</a>,
 * <a href="../actions/package-summary.html">actions</a> and
 * <a href="../formcomponents/package-summary.html">form components</a>.
 * <p>Using this class is not strictly required. In principle, you could also set values for the {@code ATTRIBUTE_*} constants in the bean descriptor
 * of an unrelated bean info class.
 */
public abstract class Install4JBeanInfo extends SimpleBeanInfo {

    /**
     * @see #setMinimumJavaVersion(String)
     */
    public static final String ATTRIBUTE_MINIMUM_JAVA_VERSION = "minimumJavaVersion";

    /**
     * @see #Install4JBeanInfo(String, String, String, boolean, boolean, Integer, Class, Class)
     */
    public static final String ATTRIBUTE_MULTIPLE_INSTANCES_SUPPORTED = "multipleInstancesSupported";

    /**
     * @see #Install4JBeanInfo(String, String, String, boolean, boolean, Integer, Class, Class)
     */
    public static final String ATTRIBUTE_INSTALLED_FILES_REQUIRED = "installedFilesRequired";

    /**
     * @see #Install4JBeanInfo(String, String, String, boolean, boolean, Integer, Class, Class)
     */
    public static final String ATTRIBUTE_BEAN_CATEGORY = "beanCategory";

    /**
     * @see #setCollapsedPropertyCategories(String[])
     */
    public static final String ATTRIBUTE_COLLAPSED_PROPERTY_CATEGORIES = "collapsedPropertyCategories";

    /**
     * @see #setEnumerationMappers(EnumerationMapper[])
     */
    public static final String ATTRIBUTE_ENUMERATION_MAPPERS = "enumerationMapper";

    /**
     * @see #setPropertyConverters(PropertyConverter[])
     */
    public static final String ATTRIBUTE_PROPERTY_CONVERTERS = "propertyConverters";

    /**
     * @see #setPersistenceDelegateMap(Map)
     */
    public static final String ATTRIBUTE_PERSISTENCE_DELEGATE_MAP = "persistenceDelegateMap";

    /**
     * @see #setCustomizerPlacement(CustomizerPlacement)
     */
    public static final String ATTRIBUTE_CUSTOMIZER_PLACEMENT = "customizerPlacement";

    /**
     * @see #setCustomizerIcon(Icon)
     */
    public static final String ATTRIBUTE_CUSTOMIZER_ICON = "customizerIcon";

    /**
     * @see #setBeanValidator(BeanValidator)
     */
    public static final String ATTRIBUTE_BEAN_VALIDATOR = "beanValidator";

    /**
     * @see #setBeanInitializer(BeanInitializer)
     */
    public static final String ATTRIBUTE_BEAN_INITIALIZER = "beanInitializer";

    /**
     * @see #Install4JBeanInfo(String, String, String, boolean, boolean, Integer, Class, Class)
     */
    public static final String ATTRIBUTE_SORT_KEY = "sortKey";

    /**
     * @see #setSequenceValidator(SequenceValidator)
     */
    public static final String ATTRIBUTE_SEQUENCE_VALIDATOR = "sequenceValidator";

    /**
     * @see #setDefaultRollbackBarrier(boolean)
     */
    public static final String ATTRIBUTE_DEFAULT_ROLLBACK_BARRIER = "defaultRollbackBarrier";

    /**
     * @see #setDefaultRollbackBarrierExitCode(int)
     */
    public static final String ATTRIBUTE_DEFAULT_ROLLBACK_BARRIER_EXIT_CODE = "defaultRollbackBarrierExitCode";

    /**
     * @see #setDefaultConditionExpression(String)
     */
    public static final String ATTRIBUTE_DEFAULT_CONDITION_EXPRESSION = "defaultConditionExpression";

    /**
     * @see #setIcons(Icon, Icon)
     */
    public static final String ATTRIBUTE_ICON_16x16 = "icon16x16";

    /**
     * @see #setIcons(Icon, Icon)
     */
    public static final String ATTRIBUTE_ICON_24x24 = "icon24x24";

    /**
     * @see #setNoticePanel(JComponent)
     */
    public static final String ATTRIBUTE_NOTICE_PANEL = "noticePanel";

    /**
     * @see #setCategorySortOrder(String[])
     */
    public static final String ATTRIBUTE_CATEGORY_SORT_ORDER = "categorySortOrder";

    /**
     * @see #setCategorySortOrderOtherAtBottom(boolean)
     */
    public static final String ATTRIBUTE_CATEGORY_SORT_ORDER_OTHER_AT_BOTTOM = "categorySortOrderOtherAtBottom";

    private List<Install4JPropertyDescriptor> propertyDescriptors = new ArrayList<>();

    private BeanDescriptor beanDescriptor;

    private int nextPropertySortKeyValue = 0;

    /*
     * When you register a custom action on the "Actions" tab of the "Installer" section in the install4j GUI,
     * the action list displays this name. By default, this method returns {@code null} which means that the
     * class name is displayed. You can override this method to display a more descriptive name. This method
     * has no effect at run time.
     * @return the descriptive name
     */

    /*
     * Determines whether an action can be placed multiple times into the action sequence at design time.
     * Since bean properties of actions can be configured at design time, you can develop a generic action
     * that is configured in different ways. The default value is {@code true}. You can override this method
     * if your action is not configurable.
     * @return if multiple instances of this action are supported
     */

    /*
     * Actions that depend on deployed files can override this method to signal that they should not be placed
     * before the "Install files" standard action or after the "Uninstall files" standard action. The default value
     * is {@code false}. This method has no effect at runtime.
     * @return if the action requires the installed files
     */

    /**
     * Constructor.<p>
     * Customizers must descend from {@code javax.swing.JComponent}. Their {@code name} property (settable with {@code JComponent#setName(String)})
     * will be displayed as the tab name next to the default "Properties" tab. You can add multiple tabs if your customizer descends from
     * {@code javax.swing.JTabbedPane}.
     * In that case, all its tabs are extracted and added next to the default "Properties" tab, and any icons that are set
     * for the single tabs will also be used in the install4j IDE.
     * If you have properties whose values are exclusively used by the customizer, you can give them an empty display name so that
     * they will not show up in the default "Properties" tab. Additional services for the customizer are available with a
     * {@link CustomizerCallback}, the default placement of the customizer can be controlled with
     * {@link #setCustomizerPlacement(CustomizerPlacement)}.
     * @param displayName the display name for the bean. This name will be displayed in the list of configured beans as well as in the bean registry dialog.
     * @param shortDescription a short description that will be displayed in the bean registry dialog in HTML format. You do not have to start the description with &lt;html&gt;, it will be prepended automatically.
     * @param category the bean category. This determines the node in the bean registry dialog where the bean will be added. If {@code null} the bean will be added at the top level.
     * @param multipleInstancesSupported if multiple instances of this bean are supported
     * @param installedFilesRequired if this bean requires installed files or not. In the installer, this means that this bean must be placed after the "Install files" action,
     *   in the uninstaller this means that this bean must be placed before the "Uninstall files" action.
     * @param sortKey an integer that will be used for determining the sort order in the bean registry dialog and the default insertion point in the
     *   list of configured beans.
     * @param beanClass the class name of the bean
     * @param customizerClass the customizer class or {@code null} if no customizer is available.
     * @see #ATTRIBUTE_BEAN_CATEGORY
     * @see #ATTRIBUTE_MULTIPLE_INSTANCES_SUPPORTED
     * @see #ATTRIBUTE_INSTALLED_FILES_REQUIRED
     * @see #ATTRIBUTE_SORT_KEY
     * @see CustomizerCallback
     */
    protected Install4JBeanInfo(String displayName, String shortDescription, String category, boolean multipleInstancesSupported, boolean installedFilesRequired, Integer sortKey, Class<? extends Bean> beanClass, Class<?> customizerClass) {
        initBeanDescriptor(beanClass, customizerClass);
        setDisplayName(displayName);
        setShortDescription(shortDescription);
        setBeanCategory(category);
        setMultipleInstancesSupported(multipleInstancesSupported);
        setInstalledFilesRequired(installedFilesRequired);
        setSortKey(sortKey);

        if (this instanceof BeanValidator) {
            setBeanValidator((BeanValidator)this);
        }

        if (this instanceof BeanInitializer) {
            setBeanInitializer((BeanInitializer)this);
        }

        if (this instanceof SequenceValidator) {
            setSequenceValidator((SequenceValidator)this);
        }
    }

    /**
     * Same as {@link #Install4JBeanInfo(String, String, String, boolean, boolean, Integer, Class, Class)} with a customizer class of {@code null}.
     */
    protected Install4JBeanInfo(String displayName, String shortDescription, String category, boolean multipleInstancesSupported, boolean installedFilesRequired, Integer sortKey, Class<? extends Bean> beanClass) {
        this(displayName, shortDescription, category, multipleInstancesSupported, installedFilesRequired, sortKey, beanClass, null);
    }

    /**
     * Add a property descriptor to be returned by {@code getPropertyDescriptors}.
     * @param propertyDescriptor the property descriptor
     */
    public void addPropertyDescriptor(Install4JPropertyDescriptor propertyDescriptor) {
        if (propertyDescriptor.getValue(Install4JPropertyDescriptor.ATTRIBUTE_SORT_KEY) == null) {
            nextPropertySortKeyValue += 100;
            propertyDescriptor.setSortKey(nextPropertySortKeyValue);
        }
        propertyDescriptors.add(propertyDescriptor);
    }

    @Override
    public PropertyDescriptor[] getPropertyDescriptors() {
        return propertyDescriptors.toArray(new PropertyDescriptor[0]);
    }

    @Override
    public BeanDescriptor getBeanDescriptor() {
        return beanDescriptor;
    }

    /**
     * Configures custom icons for the bean. The 16x16 icon will be displayed in the bean registry dialog, the
     * 24x24 icon will be displayed in the list of configured beans.
     * @param icon16x16 the 16x16 icon
     * @param icon24x24 the 32x32 icon
     * @see #ATTRIBUTE_ICON_16x16
     * @see #ATTRIBUTE_ICON_24x24
     */
    public void setIcons(Icon icon16x16, Icon icon24x24) {
        BeanDescriptor beanDescriptor = getBeanDescriptor();

        if (icon16x16 != null) {
            beanDescriptor.setValue(ATTRIBUTE_ICON_16x16, icon16x16);
        }
        if (icon24x24 != null) {
            beanDescriptor.setValue(ATTRIBUTE_ICON_24x24, icon24x24);
        }
    }

    /**
     * Sets the minimum Java version that this bean will work with. If the minimum Java version for the
     * project is less than this version, the bean cannot be added and an error message is displayed.
     * @param minimumJavaVersion the minimum Java version
     * @see #ATTRIBUTE_MINIMUM_JAVA_VERSION
     */
    public void setMinimumJavaVersion(String minimumJavaVersion) {
        if (minimumJavaVersion != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_MINIMUM_JAVA_VERSION, minimumJavaVersion);
        }
    }

    /**
     * Specifies property categories that should be collapsed by default.
     * @param collapsedCategories an array with the collapsed categories
     * @see Install4JPropertyDescriptor#setPropertyCategory(String)
     * @see #ATTRIBUTE_COLLAPSED_PROPERTY_CATEGORIES
     */
    public void setCollapsedPropertyCategories(String[] collapsedCategories) {
        if (collapsedCategories != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_COLLAPSED_PROPERTY_CATEGORIES, collapsedCategories);
        }
    }

    /**
     * Specifies enumeration mappers for properties of this bean.
     * @param enumerationMappers an array with the enumeration mappers
     * @see EnumerationMapper
     * @see #ATTRIBUTE_ENUMERATION_MAPPERS
     */
    public void setEnumerationMappers(EnumerationMapper[] enumerationMappers) {
        if (enumerationMappers != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_ENUMERATION_MAPPERS, enumerationMappers);
        }
    }

    /**
     * Specifies property converters for properties of this bean.
     * @param propertyConverters an array with the property converters
     * @see PropertyConverter
     * @see #ATTRIBUTE_PROPERTY_CONVERTERS
     */
    public void setPropertyConverters(PropertyConverter[] propertyConverters) {
        if (propertyConverters != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_PROPERTY_CONVERTERS, propertyConverters);
        }
    }

    /**
     * Specifies persistence delegates for this bean. The keys must be of class
     * {@code java.lang.Class} and the values of class {@code java.beans.PersistenceDelegate}.
     * See <a href="http://java.sun.com/products/jfc/tsc/articles/persistence4/">http://java.sun.com/products/jfc/tsc/articles/persistence4/</a>
     * for more information on persistence delegates.
     * @param persistenceDelegates the {@code java.lang.Class} -&gt; {@code java.beans.PersistenceDelegate} map
     * @see #ATTRIBUTE_PERSISTENCE_DELEGATE_MAP
     */
    public void setPersistenceDelegateMap(Map<? extends Class, ? extends PersistenceDelegate> persistenceDelegates) {
        if (persistenceDelegates != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_PERSISTENCE_DELEGATE_MAP, persistenceDelegates);
        }
    }

    /**
     * Configures the placement of the customizer.
     * @param customizerPlacement the customizer placement
     * @see #Install4JBeanInfo(String, String, String, boolean, boolean, Integer, Class, Class)
     * @see #ATTRIBUTE_CUSTOMIZER_PLACEMENT
     */
    public void setCustomizerPlacement(CustomizerPlacement customizerPlacement) {
        if (customizerPlacement != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_CUSTOMIZER_PLACEMENT, customizerPlacement);
        }
    }

    /**
     * Configures the icon of the customizer tab.
     * @param customizerIcon the icon for the customizer tab
     * @see #ATTRIBUTE_CUSTOMIZER_ICON
     */
    public void setCustomizerIcon(Icon customizerIcon) {
        if (customizerIcon != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_CUSTOMIZER_ICON, customizerIcon);
        }
    }

    /** Configures a bean validator.
     * @param beanValidator the bean validator
     * @see BeanValidator
     * @see #ATTRIBUTE_BEAN_VALIDATOR
     */
    public void setBeanValidator(BeanValidator beanValidator) {
        if (beanValidator != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_BEAN_VALIDATOR, beanValidator);
        }
    }

    /**
     * Configures a bean initializer.
     * @param beanInitializer the bean initializer.
     * @see BeanInitializer
     * @see #ATTRIBUTE_BEAN_INITIALIZER
     */
    public void setBeanInitializer(BeanInitializer beanInitializer) {
        if (beanInitializer != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_BEAN_INITIALIZER, beanInitializer);
        }
    }

    /**
     * Configures a sequence validator.
     * @param sequenceValidator the sequence validator.
     * @see SequenceValidator
     * @see #ATTRIBUTE_SEQUENCE_VALIDATOR
     */
    public void setSequenceValidator(SequenceValidator sequenceValidator) {
        if (sequenceValidator != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_SEQUENCE_VALIDATOR, sequenceValidator);
        }
    }

    /**
     * Configures if the action is a rollback barrier by default.
     * @param defaultRollbackBarrier the default rollback barrier flag
     * @see #ATTRIBUTE_DEFAULT_ROLLBACK_BARRIER
     */
    public void setDefaultRollbackBarrier(boolean defaultRollbackBarrier) {
        getBeanDescriptor().setValue(ATTRIBUTE_DEFAULT_ROLLBACK_BARRIER, defaultRollbackBarrier);
    }

    /**
     * Configures the default rollback barrier exit code.
     * Only has an effect if the "Rollback barrier" property is selected by the user.
     * @param defaultRollbackBarrierExitCode the default rollback barrier type
     * @see #ATTRIBUTE_DEFAULT_ROLLBACK_BARRIER_EXIT_CODE
     */
    public void setDefaultRollbackBarrierExitCode(int defaultRollbackBarrierExitCode) {
        getBeanDescriptor().setValue(ATTRIBUTE_DEFAULT_ROLLBACK_BARRIER_EXIT_CODE, defaultRollbackBarrierExitCode);
    }

    /**
     * Configures the default value for the "Condition expression" property of the bean.
     * @param defaultConditionExpression the default value
     * @see #ATTRIBUTE_DEFAULT_CONDITION_EXPRESSION
     */
    public void setDefaultConditionExpression(String defaultConditionExpression) {
        getBeanDescriptor().setValue(ATTRIBUTE_DEFAULT_CONDITION_EXPRESSION, defaultConditionExpression);
    }

    /**
     * Sets a panel that is displayed a notice at the top of the configuration panel. This panel cannot be used for
     * configuring the bean.
     * @param noticePanel the panel
     */
    public void setNoticePanel(JComponent noticePanel) {
        getBeanDescriptor().setValue(ATTRIBUTE_NOTICE_PANEL, noticePanel);
    }

    /**
     * Sets a sort order for property categories of this bean. By default, property categories are sorted alphabetically.
     * You can change the sort order for the categories that are passed to this method.
     * Any categories that are not present in this sort order will still be sorted alphabetically and
     * will be shown at the top of the property tree.
     * @param categories an array with all categories that should be sorted in the order that the categories should appear
     *      in the install4j IDE
     */
    public void setCategorySortOrder(String... categories) {
        if (categories != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_CATEGORY_SORT_ORDER, categories);
        }
    }

    /**
     * If set to true, categories that are not present in those passed to {@link #setCategorySortOrder(String...)}
     * are displayed at the bottom of the property table.
     * @param categorySortOrderOtherAtBottom {@code true} or {@code false}.
     */
    public void setCategorySortOrderOtherAtBottom(boolean categorySortOrderOtherAtBottom) {
        getBeanDescriptor().setValue(ATTRIBUTE_CATEGORY_SORT_ORDER_OTHER_AT_BOTTOM, categorySortOrderOtherAtBottom);
    }

    /**
     * Convenience method to retrieve the bean class specified in the constructor.
     * @return the bean class
     */
    public Class<?> getBeanClass() {
        return beanDescriptor.getBeanClass();
    }

    /**
     * Convenience method for bean validators to assert that a property is not empty.
     * A {@link BeanValidationException} with an appropriate message is thrown if the property is empty.
     * @param propertyName the name of the property that should be checked
     * @param bean the bean whose property should be checked
     * @throws BeanValidationException if the specified property is empty
     * @see #isEmpty(String, Bean)
     * @see BeanValidator
     */
    public void checkNotEmpty(String propertyName, Bean bean) throws BeanValidationException {
        if (isEmpty(propertyName, bean)) {
            PropertyDescriptor propertyDescriptor = findPropertyDescriptor(propertyName);
            String displayName = propertyDescriptor.getDisplayName();
            if (displayName == null || displayName.trim().isEmpty()) {
                throw new BeanValidationException("Please enter all required information.");
            } else {
                throw new BeanValidationException("The property \"" + displayName + "\" must not be empty.", propertyName);
            }
        }
    }

    /**
     * Convenience method for bean validators to assert that a property is not empty.
     * A {@link BeanValidationException} with the specified error message is thrown if the property is empty.
     * @param propertyName the name of the property that should be checked
     * @param errorMessage the error message
     * @param bean the bean whose property should be checked
     * @throws BeanValidationException if the specified property is empty
     * @see #isEmpty(String, Bean)
     * @see BeanValidator
     */
    public void checkNotEmpty(String propertyName, String errorMessage, Bean bean) throws BeanValidationException {
        if (isEmpty(propertyName, bean)) {
            throw new BeanValidationException(errorMessage);
        }
    }

    /**
     * Convenience method for bean validators to check if a property is empty.
     * @param propertyName the name of the property that should be checked
     * @param bean the bean whose property should be checked
     * @return {@code true} or {@code false}.
     */
    public boolean isEmpty(String propertyName, Bean bean) {
        PropertyDescriptor propertyDescriptor = findPropertyDescriptor(propertyName);
        if (propertyDescriptor == null) {
            return false;
        }

        Object value = getPropertyValue(propertyDescriptor, bean);
        if (value == null) {
            return true;
        }
        if (value.getClass().isArray()) {
            return Array.getLength(value) == 0;
        } else if (value instanceof LocalizedExternalFile) {
            return ((LocalizedExternalFile)value).getLanguageIdToExternalFile().isEmpty();
        } else if (value instanceof List) {
            return ((List)value).isEmpty();
        } else if (value instanceof Map) {
            return ((Map)value).isEmpty();
        } else if (value instanceof ScriptProperty) {
            return ((ScriptProperty)value).getValue().isEmpty();
        }
        return value.toString().trim().isEmpty();
    }

    /**
     * Convenience method for bean validators to get the property value for a property descriptor.
     * @param propertyDescriptor the property descriptor for which the value should be returned
     * @param bean the bean for which the value should be returned
     * @return the property value or {@code null} if the property cannot be found
     */
    public Object getPropertyValue(PropertyDescriptor propertyDescriptor, Bean bean) {
        if (propertyDescriptor == null || bean == null) {
            return null;
        }
        Method readMethod = propertyDescriptor.getReadMethod();
        try {
            return readMethod.invoke(bean);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * Convenience method for bean validators to get the property value for a named property.
     * @param propertyName the name of the property for which the value should be returned
     * @param bean the bean for which the value should be returned
     * @return the property value or {@code null} if the property cannot be found
     */
    public Object getPropertyValue(String propertyName, Bean bean) {
        PropertyDescriptor propertyDescriptor = findPropertyDescriptor(propertyName);
        if (propertyDescriptor == null) {
            return null;
        } else {
            return getPropertyValue(propertyDescriptor, bean);
        }
    }

    /**
     * Convenience method for bean validators to find the property descriptor for a named property.
     * @param propertyName the name of the property for which the property descriptor should be returned
     * @return the property descriptor
     */
    public PropertyDescriptor findPropertyDescriptor(String propertyName) {
        PropertyDescriptor[] propertyDescriptors = getPropertyDescriptors();
        if (propertyDescriptors == null) {
            return null;
        }
        for (PropertyDescriptor propertyDescriptor : propertyDescriptors) {
            if (Objects.equals(propertyDescriptor.getName(), propertyName)) {
                return propertyDescriptor;
            }
        }
        return null;
    }

    private void initBeanDescriptor(Class<? extends Bean> beanClass, Class<?> customizerClass) {
        beanDescriptor = new BeanDescriptor(beanClass, customizerClass);
    }

    private void setDisplayName(String displayName) {
        beanDescriptor.setDisplayName(displayName);
    }

    private void setShortDescription(String shortDescription) {
        beanDescriptor.setShortDescription(shortDescription);
    }

    private void setBeanCategory(String category) {
        if (category != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_BEAN_CATEGORY, category);
        }
    }

    private void setMultipleInstancesSupported(boolean multipleInstancesSupported) {
        getBeanDescriptor().setValue(ATTRIBUTE_MULTIPLE_INSTANCES_SUPPORTED, multipleInstancesSupported);
    }

    private void setInstalledFilesRequired(boolean installedFilesRequired) {
        getBeanDescriptor().setValue(ATTRIBUTE_INSTALLED_FILES_REQUIRED, installedFilesRequired);
    }

    private void setSortKey(Integer sortKey) {
        if (sortKey != null) {
            getBeanDescriptor().setValue(ATTRIBUTE_SORT_KEY, sortKey);
        }
    }

}
