package com.install4j.api.windows;

import com.install4j.runtime.installer.helper.comm.ExecutionContext;
import com.install4j.runtime.installer.helper.comm.HelperCommunication;
import com.install4j.runtime.installer.platform.win32.Registry;

import java.io.Serializable;

/**
 * Collection of static methods to access the Microsoft Windows registry.
 * <p>
 * If {@code RegistryRoot.HKEY_CURRENT_USER} is specified, the correct
 * result for the original user is returned even if executed in the elevated helper.
 * </p>
 * @author ej-technologies GmbH
 */
public class WinRegistry {

    /**
     * Checks if a registry key exists.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash.
     * @return {@code true} if the key exists.
     */
    public static boolean keyExists(RegistryRoot root, String keyName) {
        return keyExists(root, keyName, RegistryView.DEFAULT);
    }

    /**
     * Creates a registry key.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash.
     * @return {@code true} if successful.
     */
    public static boolean createKey(RegistryRoot root, String keyName) {
        return createKey(root, keyName, RegistryView.DEFAULT);
    }

    /**
     * Retrieves a registry value. The class of the returned object is determined
     * by the value's type.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash.
     * @param valueName the name of the registry value.
     * @return the requested value. Returns {@code null}, if the value doesn't exist. Otherwise
     * it is an instance of one of the following classes: {@code String, Integer, Long, String[], byte[], WinRegistry.ExpandString}.
     */
    public static Object getValue(RegistryRoot root, String keyName, String valueName) {
        return getValue(root, keyName, valueName, RegistryView.DEFAULT);
    }

    /**
     * Sets a registry value. The type is determined by the class of the value.
     * @param root one of the {@code RegistryRoot.*} constants
     * @param keyName the name of the key without a leading backslash.
     * @param valueName the name of the registry value.
     * @param value an instance of one of the following classes: {@code String, Integer, Long, String[], byte[], WinRegistry.ExpandString}.
     * @return {@code true} if successful.
     */
    public static boolean setValue(RegistryRoot root, String keyName, String valueName, Object value) {
        return setValue(root, keyName, valueName, value, RegistryView.DEFAULT);
    }

    /**
     * Deletes a registry value.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash.
     * @param valueName the name of the registry value.
     */
    public static void deleteValue(RegistryRoot root, String keyName, String valueName) {
        deleteValue(root, keyName, valueName, RegistryView.DEFAULT);
    }

    /**
     * Deletes a registry key.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash.
     * @param onlyIfEmpty if {@code true}, the method will delete the key only if the key contains no
     * values and no subkey.
     */
    public static void deleteKey(RegistryRoot root, String keyName, boolean onlyIfEmpty) {
        deleteKey(root, keyName, onlyIfEmpty, RegistryView.DEFAULT);
    }

    /**
     * Retrieves the subkeys of a registry key.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash.
     * @return an array with the subkeys' names. Returns an empty array if there are no subkeys and
     * {@code null} if the key doesn't exist or if an error occurs.
     */
    public static String[] getSubKeyNames(RegistryRoot root, String keyName) {
        return getSubKeyNames(root, keyName, RegistryView.DEFAULT);
    }

    /**
     * Retrieves the exiting value names of a registry key.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash.
     * @return an array with the values' names. Returns an empty array if there are no values and
     * {@code null} if the key doesn't exist or if an error occurs.
     */
    public static String[] getValueNames(RegistryRoot root, String keyName) {
        return getValueNames(root, keyName, RegistryView.DEFAULT);
    }

    /**
     * Save a subtree of the registry to a file.
     * The subtree can be restored with {@code restoreKey}.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash. This is the root of the saved subtree.
     * @param fileName the file to which the subtree should be saved.
     * @return whether the operation was successful or not.
     * @see #restoreKey(RegistryRoot, String, String)
     */
    public static boolean saveKey(RegistryRoot root, String keyName, String fileName) {
        return saveKey(root, keyName, fileName, RegistryView.DEFAULT);
    }

    /**
     * Restore a subtree of the registry from a file.
     * This method can restore files saved with the {@code saveKey} method.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash. This is the key to which the root of the saved subtree
     * will be restored.
     * @param fileName the file from which the saved subtree should be read.
     * @return whether the operation was successful or not.
     * @see #saveKey(RegistryRoot, String, String)
     */
    public static boolean restoreKey(RegistryRoot root, String keyName, String fileName) {
        return restoreKey(root, keyName, fileName, RegistryView.DEFAULT);
    }

    /**
     * Checks if a registry key exists.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash.
     * @param registryView the registry view used on 64-bit systems.
     * @return {@code true} if the key exists.
     */
    public static boolean keyExists(final RegistryRoot root, final String keyName, final RegistryView registryView) {
        if (isUnelevate(root)) {
            return HelperCommunication.getInstance().fetchBoolean(ExecutionContext.UNELEVATED, context -> keyExists(root, keyName, registryView));
        } else {
            return Registry.keyExists(root.getType(), keyName, registryView.getUsedType());
        }
    }

    /**
     * Creates a registry key.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash.
     * @param registryView the registry view used on 64-bit systems.
     * @return {@code true} if successful.
     */
    public static boolean createKey(final RegistryRoot root, final String keyName, final RegistryView registryView) {
        if (isUnelevate(root)) {
            return HelperCommunication.getInstance().fetchBoolean(ExecutionContext.UNELEVATED, context -> createKey(root, keyName, registryView));
        } else {
            return Registry.createKey(root.getType(), keyName, registryView.getUsedType());
        }
    }

    /**
     * Retrieves a registry value. The class of the returned object is determined
     * by the value's type.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash.
     * @param valueName the name of the registry value.
     * @param registryView the registry view used on 64-bit systems.
     * @return the requested value. Returns {@code null}, if the value doesn't exist. Otherwise
     * it is an instance of one of the following classes: {@code String, Integer, Long, String[], byte[], WinRegistry.ExpandString}.
     */
    public static Object getValue(final RegistryRoot root, final String keyName, final String valueName, final RegistryView registryView) {
        if (isUnelevate(root)) {
            return HelperCommunication.getInstance().fetchObject(ExecutionContext.UNELEVATED, context -> getValue(root, keyName, valueName, registryView));
        } else {
            return Registry.getValue(root.getType(), keyName, valueName, registryView.getUsedType());
        }
    }

    /**
     * Sets a registry value. The type is determined by the class of the value.
     * @param root one of the {@code RegistryRoot.*} constants
     * @param keyName the name of the key without a leading backslash.
     * @param valueName the name of the registry value.
     * @param value an instance of one of the following classes: {@code String, Integer, Long, String[], byte[], WinRegistry.ExpandString}.
     * @param registryView the registry view used on 64-bit systems.
     * @return {@code true} if successful.
     */
    public static boolean setValue(final RegistryRoot root, final String keyName, final String valueName, final Object value, final RegistryView registryView) {
        if (isUnelevate(root)) {
            return HelperCommunication.getInstance().fetchBoolean(ExecutionContext.UNELEVATED, context -> setValue(root, keyName, valueName, value, registryView));
        } else {
            Object usedValue = value instanceof ExpandString ? new Registry.ExpandStringImpl(((ExpandString)value).getValue()) : value;
            return Registry.setValue(root.getType(), keyName, valueName, usedValue, registryView.getUsedType());
        }
    }

    /**
     * Deletes a registry value.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash.
     * @param valueName the name of the registry value.
     * @param registryView the registry view used on 64-bit systems.
     */
    public static void deleteValue(final RegistryRoot root, final String keyName, final String valueName, final RegistryView registryView) {
        if (isUnelevate(root)) {
            HelperCommunication.getInstance().runAction(ExecutionContext.UNELEVATED, context -> deleteValue(root, keyName, valueName, registryView));
        } else {
            Registry.deleteValue(root.getType(), keyName, valueName, registryView.getUsedType());
        }
    }

    /**
     * Deletes a registry key.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash.
     * @param onlyIfEmpty if {@code true}, the method will delete the key only if the key contains no
     * values and no subkey.
     * @param registryView the registry view used on 64-bit systems.
     */
    public static void deleteKey(final RegistryRoot root, final String keyName, final boolean onlyIfEmpty, final RegistryView registryView) {
        if (isUnelevate(root)) {
            HelperCommunication.getInstance().runAction(ExecutionContext.UNELEVATED, context -> deleteKey(root, keyName, onlyIfEmpty, registryView));
        } else {
            Registry.deleteKey(root.getType(), keyName, onlyIfEmpty, registryView.getUsedType());
        }
    }

    /**
     * Retrieves the subkeys of a registry key.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash.
     * @return an array with the subkeys' names. Returns an empty array if there are no subkeys and
     * {@code null} if the key doesn't exist or if an error occurs.
     * @param registryView the registry view used on 64-bit systems.
     */
    public static String[] getSubKeyNames(final RegistryRoot root, final String keyName, final RegistryView registryView) {
        if (isUnelevate(root)) {
            return HelperCommunication.getInstance().fetchObject(ExecutionContext.UNELEVATED, context -> getSubKeyNames(root, keyName, registryView));
        } else {
            return Registry.enumSubKeys(root.getType(), keyName, registryView.getUsedType());
        }
    }

    /**
     * Retrieves the exiting value names of a registry key.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash.
     * @return an array with the values' names. Returns an empty array if there are no values and
     * @param registryView the registry view used on 64-bit systems.
     * {@code null} if the key doesn't exist or if an error occurs.
     */
    public static String[] getValueNames(final RegistryRoot root, final String keyName, final RegistryView registryView) {
        if (isUnelevate(root)) {
            return HelperCommunication.getInstance().fetchObject(ExecutionContext.UNELEVATED, context -> getValueNames(root, keyName, registryView));
        } else {
            return Registry.enumValues(root.getType(), keyName, registryView.getUsedType());
        }
    }

    /**
     * Save a subtree of the registry to a file.
     * The subtree can be restored with {@code restoreKey}.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash. This is the root of the saved subtree.
     * @param fileName the file to which the subtree should be saved.
     * @param registryView the registry view used on 64-bit systems.
     * @return whether the operation was successful or not.
     * @see #restoreKey(RegistryRoot, String, String)
     */
    public static boolean saveKey(final RegistryRoot root, final String keyName, final String fileName, final RegistryView registryView) {
        if (isUnelevate(root)) {
            return HelperCommunication.getInstance().fetchBoolean(ExecutionContext.UNELEVATED, context -> saveKey(root, keyName, fileName, registryView));
        } else {
            return Registry.saveKey(root.getType(), keyName, fileName, registryView.getUsedType());
        }
    }

    /**
     * Restore a subtree of the registry from a file.
     * This method can restore files saved with the {@code saveKey} method.
     * @param root one of the {@code RegistryRoot.*} constants.
     * @param keyName the name of the key without a leading backslash. This is the key to which the root of the saved subtree
     * will be restored.
     * @param fileName the file from which the saved subtree should be read.
     * @param registryView the registry view used on 64-bit systems.
     * @return whether the operation was successful or not.
     * @see #saveKey(RegistryRoot, String, String)
     */
    public static boolean restoreKey(final RegistryRoot root, final String keyName, final String fileName, final RegistryView registryView) {
        if (isUnelevate(root)) {
            return HelperCommunication.getInstance().fetchBoolean(ExecutionContext.UNELEVATED, context -> restoreKey(root, keyName, fileName, registryView));
        } else {
            return Registry.restoreKey(root.getType(), keyName, fileName, registryView.getUsedType());
        }
    }

    private static boolean isUnelevate(RegistryRoot root) {
        return HelperCommunication.getInstance().isElevatedHelper() && root == RegistryRoot.HKEY_CURRENT_USER;
    }

    /**
     * Class to represent Strings with type REG_EXPAND_SZ.
     */
    public static class ExpandString implements Serializable {
        private String value;

        /**
         * Constructor.
         * @param value the string value
         */
        public ExpandString(String value) {
            this.value = value;
        }

        /**
         * Get the string value.
         * @return the string value
         */
        public String getValue() {
            return value;
        }

        @Override
        public String toString() {
            return value;
        }
    }


    private WinRegistry() {
    }
}
