package com.install4j.api.update;

import com.install4j.api.beans.ErrorHandlingCallback;
import com.install4j.api.beans.ErrorHandlingMode;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * An update check request for {@link UpdateChecker#getUpdateDescriptor(UpdateCheckRequest)}.
 * Only the {@code urlSpec} is mandatory, all other options have a default value that is documented in the setters.
 * This class is intended to be used in a fluent API invocation style:
 * <pre>{@code
 * UpdateDescriptor updateDescriptor = UpdateChecker.getUpdateDescriptor(
 *      new UpdateCheckRequest("http://test.com/updates.xml").
 *          applicationDisplayMode(ApplicationDisplayMode.CONSOLE).
 *          askForProxy(false).
 *          connectTimeout(50000)
 *      )
 * );
 * }</pre>
 *
 * @author ej-technologies GmbH
 */
public class UpdateCheckRequest {
    private String urlSpec;
    private ApplicationDisplayMode applicationDisplayMode = ApplicationDisplayMode.GUI;
    private ErrorHandlingCallback errorHandlingCallback = null;
    private Map<String, List<String>> requestHeaders = new LinkedHashMap<>();
    private int connectTimeout = 10000;
    private int readTimeout = 10000;
    private boolean acceptAllCertificates = false;

    /**
     * Constructs an update check request instance.
     *
     * @param urlSpec the URL from which the {@code updates.xml} file can be downloaded.
     */
    public UpdateCheckRequest(String urlSpec) {
        urlSpec(urlSpec);
    }

    /**
     * Returns the value set with {@link #urlSpec(String)}
     *
     * @return the value
     */
    public String getUrlSpec() {
        return urlSpec;
    }

    /**
     * Sets the URL from which the {@code updates.xml} file can be downloaded.
     *
     * @param urlSpec the url
     * @return this instance
     */
    public UpdateCheckRequest urlSpec(String urlSpec) {
        this.urlSpec = urlSpec;
        return this;
    }

    /**
     * Returns the value set with {@link #applicationDisplayMode(ApplicationDisplayMode)}
     * The default value is {@link ApplicationDisplayMode#GUI}.
     *
     * @return the value
     */
    public ApplicationDisplayMode getApplicationDisplayMode() {
        return applicationDisplayMode;
    }

    /**
     * Sets the display mode that should be used if proxy information should be required
     */
    public UpdateCheckRequest applicationDisplayMode(ApplicationDisplayMode applicationDisplayMode) {
        this.applicationDisplayMode = applicationDisplayMode;
        return this;
    }

    /**
     * Returns the value set with {@link #errorHandlingCallback(ErrorHandlingCallback)}
     *
     * @return the value
     */
    public ErrorHandlingCallback getErrorHandlingCallback() {
        return errorHandlingCallback;
    }

    /**
     * Sets the callback for handling network errors. If you return {@link ErrorHandlingMode#IGNORE},
     * the regular proxy or failure handling will proceed, if you return {@link ErrorHandlingMode#CANCEL}
     * the action will fail immediately. If you can take corrective action in the script, you can
     * return {@link ErrorHandlingMode#RETRY} to make the same HTTP request again. However, you have to
     * take special care not to enter an infinite loop. Typically, there should be user input before you
     * retry and the user should be given the option to cancel.
     * <p>The script is only executed for actual network failures, and not if the server or the proxy
     * connection require authentication.
     * <p>The default value is {@code null}.

     *
     * @param errorHandlingCallback the callback
     * @return this instance
     */
    public UpdateCheckRequest errorHandlingCallback(ErrorHandlingCallback errorHandlingCallback) {
        this.errorHandlingCallback = errorHandlingCallback;
        return this;
    }

    @Deprecated
    public boolean isAskForProxy() {
        return false;
    }

    @Deprecated
    public UpdateCheckRequest askForProxy(boolean askForProxy) {
        return this;
    }

    @Deprecated
    public boolean isShowProxyOnErrorCode() {
        return false;
    }

    @Deprecated
    public UpdateCheckRequest showProxyOnErrorCode(boolean showProxyOnErrorCode) {
        return this;
    }

    /**
     * Returns the value set with {@link #requestHeaders(Map)}. The default map is non-null and modifiable.
     *
     * @return the value
     */
    public Map<String, List<String>> getRequestHeaders() {
        return requestHeaders;
    }

    /**
     * Sets request headers that should be set for the HTTP connection.
     * The default value is an empty map.
     *
     * @param requestHeaders the request headers as a map of key-value pairs
     * @return this instance
     */
    public UpdateCheckRequest requestHeaders(Map<String, List<String>> requestHeaders) {
        this.requestHeaders = requestHeaders;
        return this;
    }

    /**
     * Returns the value set with {@link #connectTimeout(int)}
     * The default value is 10000 ms.
     *
     * @return the value
     */
    public int getConnectTimeout() {
        return connectTimeout;
    }

    /**
     * Sets the connect timeout in milliseconds for the HTTP connection.
     *
     * @param connectTimeout the value in milliseconds
     * @return this instance
     */
    public UpdateCheckRequest connectTimeout(int connectTimeout) {
        this.connectTimeout = connectTimeout;
        return this;
    }

    /**
     * Returns the value set with {@link #readTimeout(int)}
     * The default value is 10000 ms.
     *
     * @return the value
     */
    public int getReadTimeout() {
        return readTimeout;
    }

    /**
     * Sets the read timeout in milliseconds for the HTTP connection.
     *
     * @param readTimeout the value in milliseconds
     * @return this instance
     */
    public UpdateCheckRequest readTimeout(int readTimeout) {
        this.readTimeout = readTimeout;
        return this;
    }

    /**
     * Returns the value set with {@link #acceptAllCertificates(boolean)}
     *
     * @return the value
     */
    public boolean isAcceptAllCertificates() {
        return acceptAllCertificates;
    }

    /**
     * Sets if an HTTPS connection should accept invalid certificates (not recommended for production).
     * The default value is {@code false}.
     *
     * @param acceptAllCertificates the value
     * @return this instance
     */
    public UpdateCheckRequest acceptAllCertificates(boolean acceptAllCertificates) {
        this.acceptAllCertificates = acceptAllCertificates;
        return this;
    }
}
