package com.install4j.api.formcomponents;

import com.install4j.api.beans.ReplacementMode;
import com.install4j.api.beans.UndefinedVariableException;
import com.install4j.api.beans.VariableErrorHandling;
import com.install4j.api.beans.VariableErrorHandlingDescriptor;
import com.install4j.api.context.NotSupportedInElevationException;
import com.install4j.api.context.UserCanceledException;
import com.install4j.api.screens.Console;
import com.install4j.api.screens.FormPanelContainer;
import com.install4j.api.screens.Screen;

import javax.swing.*;
import javax.swing.event.ChangeListener;
import java.util.List;
import java.util.Set;

/**
 * The form environment gives access to form components in a form panel as well as to other design time
 * or runtime services related to form components.
 * <p>It provides access to the {@link ComponentTuple} of a form component
 * as well as to the IDs that the framework assigns to form components.
 * Furthermore, the form environment contains utility methods that can be used at design time.
 * This class is passed to the containing screen as well as to every form component.
 *
 * @author ej-technologies GmbH
 * @see FormPanelContainer#setFormPanel(JPanel, FormEnvironment)
 * @see FormComponent#setFormEnvironment(FormEnvironment)
 */
public interface FormEnvironment {

    /**
     * Returns all form components that are contained in the same form.
     * Note that the {@code setFormEnvironment} is called twice at runtime
     * and that only the second invocation passes a {@code FormEnvironment} that
     * returns all form components when calling this method.<p>
     * If you need to access layout groups, please see
     * {@link #getFormComponentTree()} instead.
     * @return an array of all form components or the empty array if the form is not fully initialized
     * @see FormComponent#setFormEnvironment(FormEnvironment)
     */
    FormComponent[] getFormComponents();

    /**
     * Returns the top level of the form component tree. The list contains
     * objects of two different classes: {@link FormComponent} and {@link LayoutGroup}.
     * Deeper levels can be accessed with {@link LayoutGroup#getChildren()} on the LayoutGroup
     * objects.
     * @return a list with the top level form components and layout groups
     */
    List<?> getFormComponentTree();

    /**
     * Get the ID that the framework associates with a form component.
     * When form components refer to other form components and wish to transfer
     * those associations between design time and runtime, they have to store a
     * IDs rather than the form components themselves. The translation between IDs
     * and form components is provided by the form environment.
     * <p>
     * The ID can be transformed back to the form component with the
     * {@code getFormComponentById} method.
     * @param formComponent the form component whose ID should be returned
     * @return the ID
     * @see #getFormComponentById(String)
     */
    String getId(FormComponent formComponent);

    /**
     * Get all form components for a given class or interface.
     *
     * @param formComponentClass the class of the form component.
     * @return the form components or an empty array if no form components of this class could be found.
     */
    <T> T[] getFormComponents(Class<T> formComponentClass);

    /**
     * Get the form component for a given class or interface. If multiple form components of the same
     * class are registered, the first occurrence is returned. In that case,
     * {@link #getFormComponents(Class)} should be used.
     *
     * @param formComponentClass the class of the screen.
     * @return the form component or {@code null} if no form component of this class could be found.
     * @see #getFormComponents(Class)
     */
    <T> T getFirstFormComponent(Class<T> formComponentClass) throws NotSupportedInElevationException;

    /**
     * Get the form component for a specified ID. See
     * {@link #getId(FormComponent)} method for more details.
     * @param id the ID of the form component
     * @return the form component or {@code null} if no such ID can be found.
     */
    FormComponent getFormComponentById(String id);

    /**
     * Get the layout group of a certain ID. See
     * {@link LayoutGroup#getId()} for more details.
     * @param id the ID of the layout group
     * @return the layout group or {@code null} if no such ID can be found.
     */
    LayoutGroup getLayoutGroupById(String id);

    /**
     * Get the {@code ComponentTuple} associated with a form component.
     * @param formComponent the form component whose {@code ComponentTuple} should be returned
     * @return the {@code ComponentTuple}
     */
    ComponentTuple getComponentTuple(FormComponent formComponent);

    /**
     * Scroll the form panel so that the specified form component becomes visible.
     * A form panel has a scroll panel only if the {@link FormPanelContainer#isScrollFormPanel()} of the parent
     * {@link FormPanelContainer} returns {@code true}. This is the default setting for empty form screens.
     * @param formComponent the form component that should be shown
     */
    void scrollComponentToVisible(FormComponent formComponent);

    /**
     * Handle console mode for all form components in a form. This method
     * has to be called at some point in the {@code handleConsole} method
     * of the containing screen, otherwise the {@code handleConsole} method of the
     * {@code FormComponent} will not be called.
     * @param console the {@code Console} object as passed to the {@code handleConsole} method of the containing screen.
     * @return whether the installer should continue or not. This value should be passed on in the
     * {@code handleConsole} method of the containing screen which is the only place where this method should be called.
     * @throws UserCanceledException if the user cancels a question or notice. This exception can be passed on to the framework.
     * @see FormComponent#handleConsole(Console)
     * @see Screen#handleConsole(Console)
     */
    boolean handleConsole(Console console) throws UserCanceledException;

    /**
     * Returns whether the form component has been instantiated at design time.
     * At design time, the context passed to a form component is {@code null}.
     * @return {@code true} or {@code false}.
     */
    boolean isDesignTime();

    /**
     * If your form component would like to be notified when the list of form components changes at design time,
     * you can add a change listener here. Only a weak reference to the listener is held,
     * so the garbage collection of your form component is not influenced.
     * <p>This method has no effect at runtime.
     * @param changeListener the listener
     */
    void addDesignTimeChangeListener(ChangeListener changeListener);

    /**
     * Remove a change listener added with {@code addDesignTimeChangeListener}.
     * <p>This method has no effect at runtime.
     * @param changeListener the listener
     */
    void removeDesignTimeChangeListener(ChangeListener changeListener);

    /**
     * At design time, the install4j GUI allows the user to name instances of form components in a form.
     * You can retrieve that name with this method.
     * @param formComponent the form component whose name should be returned.
     * @return name the design time name or {@code null} if called at runtime
     */
    String getDesignTimeName(FormComponent formComponent);

    /**
     * At design time, the install4j GUI allows the user to name instances of layout groups in a form.
     * You can retrieve that name with this method.
     * @param layoutGroup the layout group whose name should be returned.
     * @return name the design time name or {@code null} if called at runtime
     */
    String getDesignTimeName(LayoutGroup layoutGroup);

    /**
     * Get the form screen associated with this form environment.
     * @return the screen
     */
    Screen getScreen();

    /**
     * All form components whose "Reset initialization on previous" property is selected,
     * will be initialized again <b>when the screen is shown the next time</b>.
     * Usually, initialization is only reset when the user presses the "Back button" on the
     * form screen. If you implement a loop and want to reset the initialization, you
     * have to call this method. The following code retrieves a form screen by ID and resets its form components:
     * <pre>{@code
     * String screenId = ...
     * // must be a form screen for the cast to work
     * FormPanelContainer formPanelContainer = (FormPanelContainer)context.getScreenById(id);
     * formPanelContainer.getFormEnvironment().resetFormComponents()
     * }</pre>
     */
    void resetFormComponents();

    /**
     * All form components, including those that do not have their "Reset initialization on previous" property selected,
     * are initialized again. This is useful for implementing a "Reset" button that brings the form to its original state.
     */
    void reinitializeFormComponents();

    /**
     * All form components will be validated and their variables will be saved.
     * This is what happens automatically when the user clicks "Next" just before the validation expression
     * of the screen is evaluated. If you need to save the form components to variables while the screen is being shown,
     * you can use this method. For example, a button form component might require some variables of other form components
     * to be set to the currently entered values.
     *
     * @return if the validation was successful. If {@code false}, only form components up to the offending form component
     * are saved.
     */
    boolean saveFormComponents();

    /**
     * Makes the parent group of a form component visible. This is useful if a form component is located beneath a
     * tabbed pane group, and the currently displayed tab is different from the parent tab of the form component.
     * @param formComponent the form component whose parent group should be made visible.
     */
    void makeParentGroupVisible(FormComponent formComponent);

    /**
     * Set text with unresolved variables on a JLabel or JTextComponent.
     * Whenever one of the installer or form variables changes, the text is updated automatically.
     * @param textWithVariables the text with the unresolved variables.
     * @param componentWithText a JLabel or a JTextComponent
     */
    void bindTextWithVariables(String textWithVariables, JComponent componentWithText);

    /**
     * Get the value of a form variable.
     * Form variables are only accessible in the current form and have their own namespace.
     * They can be used to coordinate between different form components or between a screen and its form components.
     * @param variableName the name of the variable.
     * @return the variable value or {@code null} if the form variable with the specified name was not defined.
     * @see #setFormVariable(String, Object)
     */
    Object getFormVariable(String variableName);

    /**
     * Set a form variable.
     * @param variableName the form variable name
     * @param value the variable value
     * @see #getFormVariable(String) for more information
     */
    void setFormVariable(String variableName, Object value);

    /**
     * Get the names of all defined form variables.
     *
     * @return a set of variable names of type {@code java.lang.String}.
     * @see #getFormVariable(String)
     */
    Set<String> getFormVariableNames();

    /**
     * Replace all form variables, installer variables and localization keys in a string.
     * This method is the same as {@link com.install4j.api.beans.AbstractBean#replaceVariables(String, ReplacementMode, VariableErrorHandlingDescriptor)}
     * only that it also replaces all form variables before replacing other variable types.
     * There are several convenience methods named {@code replaceFormVariables} on {@link AbstractFormComponent} that
     * call this method.
     * <p>Note: compiler variables in the project are replaced at compile time. Compiler variables in external files,
     * such as a custom localization file, are not replaced at a compile time.
     * @param value the original string
     * @param replacementMode the replacement mode
     * @param errorHandlingDescriptor describes how missing variables should be treated for each variable type
     * @return the string with all variables replaced.
     * @throws UndefinedVariableException if a variable name cannot be found and the error handling is set to
     * {@link VariableErrorHandling#EXCEPTION} for the variable type.
     */
    String replaceFormVariables(String value, ReplacementMode replacementMode, VariableErrorHandlingDescriptor errorHandlingDescriptor) throws UndefinedVariableException;

    /**
     * Check if the form panel is fully initialized.
     * If that is not the case, accessing other form components will not be possible.
     * @return if the construction of the form panel has been completed
     */
    boolean isInitialized();

    /**
     * Get the widget style handler for applying widget styles to UI components.
     * @return the widget style handler
     */
    WidgetStyleHandler getWidgetStyleHandler();
}
