package com.install4j.api.context;

import java.util.Date;

/**
 * An object that contains options for the installation and uninstallation of a file.
 * Objects of this class have to be used as parameters for the
 * {@code installFile(...)} methods of the {@code InstallerContext}.
 * @author ej-technologies GmbH
 * @see InstallerContext#installFile(java.io.File, java.io.File, FileOptions)
 * @see InstallerContext#installFile(java.io.File, java.io.File, FileOptions, ProgressInterface, int, int)
 */
public class FileOptions {

    /**
     * The default file mode on Unix and macOS ("644").
     */
    public static final String DEFAULT_MODE = "644";

    private long fileTime = new Date().getTime();
    private String mode = DEFAULT_MODE;
    private OverwriteMode overwriteMode = OverwriteMode.ALWAYS_ASK_EXCEPT_FOR_UPDATE;

    private boolean shared = false;
    private UninstallMode uninstallMode = UninstallMode.IF_CREATED;

    private boolean delayIfNecessary = false;

    /**
     * Initializes a FileOptions instance. This is the same as calling
     * {@link #FileOptions(long, String, OverwriteMode, boolean, UninstallMode) FileOptions(new Date().getTime(),
     * DEFAULT_MODE, OverwriteMode.ALWAYS_ASK_EXCEPT_FOR_UPDATE, false, UninstallMode.IF_CREATED)}.
     */
    public FileOptions() {
    }

    /**
     * Initializes a FileOptions instance. This is the same as calling
     * {@link #FileOptions(long, String, OverwriteMode, boolean, UninstallMode) FileOptions(new Date().getTime(),
     * mode, overwriteMode, shared, UninstallMode.IF_CREATED)}.
     * @param mode the mode for the destination file (e.g. "644"). This has no effect on Windows.
     * @param overwriteMode how an existing destination file should be handled.
     * @param shared the file will be installed as shared on Windows. This means that a reference counter will
     * ensure that the file won't be uninstalled if it is also used by another program. This is highly
     * recommended for files that will be installed in Windows system directories.
     */
    public FileOptions(String mode, OverwriteMode overwriteMode, boolean shared) {
        this.mode = mode;
        this.overwriteMode = overwriteMode;
        this.shared = shared;
    }

    /**
     * Initializes a FileOptions instance. This is the same as calling
     * {@link #FileOptions(long, String, OverwriteMode, boolean, UninstallMode) FileOptions(fileTime,
     * DEFAULT_MODE, OVERWRITE_ALWAYS_ASK, false, UninstallMode.IF_CREATED)}.
     * @param fileTime the "last modified" time the file will be set to.
     */
    public FileOptions(long fileTime) {
        setFileTime(fileTime);
    }

    /**
     * Initializes a FileOptions instance. This is the same as calling
     * {@link #FileOptions(long, String, OverwriteMode, boolean, UninstallMode) FileOptions(fileTime,
     * DEFAULT_MODE, overwrite, false, UninstallMode.IF_CREATED)}.
     * @param fileTime the "last modified" time the file will be set to.
     * @param overwriteMode how an existing destination file should be handled.
     */
    public FileOptions(long fileTime, OverwriteMode overwriteMode) {
        setFileTime(fileTime);
        this.overwriteMode = overwriteMode;
    }

    /**
     * Initializes a FileOptions instance.
     * @param fileTime the "last modified" time the file will be set to.
     * @param mode the mode for the destination file (e.g. "644"). This has no effect on Windows.
     * @param overwriteMode how an existing destination file should be handled.
     * @param shared the file will be installed as shared on Windows. This means that a reference counter will
     * ensure that the file won't be uninstalled if it is also used by another program. This is highly
     * recommended for files that will be installed in Windows system directories.
     * @param uninstallMode the behavior for uninstallation.
     */
    public FileOptions(long fileTime, String mode, OverwriteMode overwriteMode, boolean shared, UninstallMode uninstallMode) {
        setFileTime(fileTime);
        this.mode = mode;
        this.overwriteMode = overwriteMode;
        this.shared = shared;
        this.uninstallMode = uninstallMode;
    }

    /**
     * Initializes a FileOptions instance.
     * @param fileTime the "last modified" time the file will be set to.
     * @param mode the mode for the destination file (e.g. "644"). This has no effect on Windows.
     * @param overwriteMode how an existing destination file should be handled.
     * @param shared the file will be installed as shared on Windows. This means that a reference counter will
     * ensure that the file won't be uninstalled if it is also used by another program. This is highly
     * recommended for files that will be installed in Windows system directories.
     * @param delayIfNecessary whether the operation should be delayed until reboot on Windows if necessary.
     * @param uninstallMode the behavior for uninstallation.
     */
    public FileOptions(long fileTime, String mode, OverwriteMode overwriteMode, boolean shared, boolean delayIfNecessary, UninstallMode uninstallMode) {
        setFileTime(fileTime);
        this.mode = mode;
        this.overwriteMode = overwriteMode;
        this.shared = shared;
        this.delayIfNecessary = delayIfNecessary;
        this.uninstallMode = uninstallMode;
    }

    /**
     * Initializes a FileOptions instance. This is the same as calling
     * {@link #FileOptions(long, String, OverwriteMode, boolean, UninstallMode) FileOptions(new Date().getTime(),
     * DEFAULT_MODE, overwriteMode, false, uninstallMode)}.
     * @param overwriteMode how an existing destination file should be handled.
     * @param uninstallMode the behavior for uninstallation.
     */
    public FileOptions(OverwriteMode overwriteMode, UninstallMode uninstallMode) {
        this.overwriteMode = overwriteMode;
        this.uninstallMode = uninstallMode;
    }

    /**
     * Returns the file time. This is the same time format as in {@code java.io.File.lastModified()}.
     * @return the file time.
     */
    public long getFileTime() {
        return fileTime;
    }

    /**
     * Sets the file time. This is the same time format as in {@code java.io.File.lastModified()}.
     * @param fileTime the new file time.
     */
    public void setFileTime(long fileTime) {
        if (fileTime > 0) {
            this.fileTime = fileTime;
        }
    }

    /**
     * Returns the unix access mode.
     * @return the mode as String (e.g. "644")
     */
    public String getMode() {
        return mode;
    }

    /**
     * Sets the unix file mode as an integer.
     * For specifying integer file modes, you can use octal numbers by prefixing
     * a "0".
     * @param intMode the mode as an integer (example: 0664).
     */
    public void setMode(int intMode) {
        //noinspection OctalInteger
        mode = Integer.toOctalString(intMode & 0777);
    }

    /**
     * Sets the unix file mode as an octal string.
     * @param mode the mode as an octal string (example: "664").
     */
    public void setMode(String mode) {
        this.mode = mode;
    }

    /**
     * Returns the overwrite mode.
     * @return one of the OverwriteMode.* constants.
     */
    public OverwriteMode getOverwriteMode() {
        return overwriteMode;
    }

    /**
     * Sets the overwrite mode.
     * @param overwriteMode one of the OverwriteMode.* constants.
     */
    public void setOverwriteMode(OverwriteMode overwriteMode) {
        this.overwriteMode = overwriteMode;
    }

    /**
     * Returns whether the file will be installed as a shared file on Windows.
     * @return {@code true} or {@code false}.
     */
    public boolean isShared() {
        return shared;
    }

    /**
     * Sets whether the file should be installed as shared on Windows.
     * @param shared {@code true} or {@code false}
     */
    public void setShared(boolean shared) {
        this.shared = shared;
    }

    /**
     * Returns the uninstallation mode.
     * @return one of the UninstallMode.* constants.
     */
    public UninstallMode getUninstallMode() {
        return uninstallMode;
    }

    /**
     * Sets whether the file should be uninstalled.
      @param uninstallMode one of the UninstallMode.* constants.
     */
    public void setUninstallMode(UninstallMode uninstallMode) {
        this.uninstallMode = uninstallMode;
    }

    /**
     * Returns whether the operation should be delayed until reboot on Windows if necessary.
     * @return {@code true} or {@code false}.
     */
    public boolean isDelayIfNecessary() {
        return delayIfNecessary;
    }

    /**
     * Sets whether the operation should be delayed until reboot on Windows if necessary.
     * @param delayIfNecessary {@code true} or {@code false}.
     */
    public void setDelayIfNecessary(boolean delayIfNecessary) {
        this.delayIfNecessary = delayIfNecessary;
    }

    @Override
    public String toString() {
        return "fileTime: " + new Date(fileTime) + ", mode: " + mode + ", overwriteMode: " + overwriteMode + ", shared: " + shared + ", uninstallMode: " + uninstallMode + ", delayIfNecessary: " + delayIfNecessary;
    }
}
