package com.install4j.jdk.spi;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.util.Collection;
import java.util.List;

/**
 * Entry point for the JDK provider. Add a text file named {@code META-INF/services/com.install4j.jdk.spi.JdkProvider}
 * to the JAR file containing the JDK provider and put the name of the implementation class in it. If the JAR file is
 * added to the class path of install4j, the JDK provider will be picked up automatically. The class path of install4j
 * can be changed by adding the line
 * <pre>{@code
 * -classpath/a <path to JAR file>
 * }</pre>
 * to the {@code bin/install4j.vmoptions} file.
 */
public interface JdkProvider {
    /**
     * A unique ID for the provider.
     */
    @NotNull
    String getId();

    /**
     * The display name for the provider. By default, this is the same as the ID.
     */
    @NotNull
    default String getDisplayName() {
        return getId();
    }

    /**
     * Returns all releases. This is a tree of category and release nodes that is displayed to the user.
     *
     * @see JdkReleaseNode
     * @see JdkCategoryNode
     */
    @NotNull
    Collection<? extends JdkNode> getReleases();

    /**
     * Find a release from a release config key. The config key is returned by {@link JdkReleaseNode#getConfigKey()}.
     * Null if not found.
     */
    @Nullable
    JdkReleaseNode getByConfigKey(@NotNull String configKey);

    /**
     * Find the latest release from a category key that also has support for the given platforms.
     * The category key is returned by {@link JdkCategoryNode#getCategoryKey()}.
     * Null if the category or no matching release could be found.
     */
    @Nullable
    JdkReleaseNode getLatestByCategoryKey(@NotNull String categoryKey, @NotNull Collection<String> requiredPlatforms);

    /**
     * Get all platforms that are available from releases the category with the given category key.
     * This is used for suggesting platforms for Linux/Unix media files.
     * Null if the category could not be found.
     */
    @Nullable
    List<String> getPlatformsByCategoryKey(@NotNull String categoryKey);

    /**
     * Remove a common prefix directory from the relative paths inside a JDK bundle. By default, nothing is removed.
     */
    @NotNull
    default File removeArchivePrefix(@NotNull File relativeFile, @NotNull String release, @NotNull String platform) {
        return relativeFile;
    }

    /**
     * Returns the common prefix directory for the directory layout of the JDK. By default, this handles the {@code Contents/Home/}
     * prefix for JDKs on macOS.
     */
    @NotNull
    default String detectDirectoryLayoutPrefix(@NotNull String platform, @NotNull String release, @NotNull File bundleFile) {
        return platform.startsWith(CommonOperatingSystems.MACOS) ? "Contents/Home/" : "";
    }
}
