package com.install4j.api.formcomponents;

import com.install4j.api.beans.AbstractBean;
import com.install4j.api.beans.ReplacementMode;
import com.install4j.api.beans.UndefinedVariableException;
import com.install4j.api.beans.VariableErrorHandlingDescriptor;
import com.install4j.api.context.Context;
import com.install4j.api.context.UserCanceledException;
import com.install4j.api.screens.Console;
import com.install4j.runtime.beans.formcomponents.FormEnvironmentContainer;
import com.install4j.runtime.beans.formcomponents.FormEnvironmentImpl;
import com.install4j.runtime.installer.InstallerVariables;
import com.install4j.runtime.installer.VariableReplacer;
import com.install4j.runtime.installer.frontend.FormPanel;
import com.install4j.runtime.installer.frontend.ScreenEnvelope;

import javax.swing.*;
import java.awt.*;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Abstract base class for form components.
 * All methods that have a reasonable default answer are overridden in the class.
 * This class saves the context and the form environment that are set by the framework and provides getters for them.
 * In addition, the "enabled" and "visible" properties are handled by invoking the corresponding setters on all
 * components in the {@code ComponentTuple} of the form component.
 *
 * @author ej-technologies GmbH
 */
public abstract class AbstractFormComponent extends AbstractBean implements FormComponent, FormEnvironmentContainer {

    public static final Pattern MNEMONIC_PATTERN = Pattern.compile("&(\\w)");

    public static String getTextWithoutMnemonics(String text) {
        return MNEMONIC_PATTERN.matcher(text).replaceFirst("$1");
    }

    public static int getMnemonicCharIndex(String text) {
        Matcher matcher = MNEMONIC_PATTERN.matcher(text);
        if (matcher.find()) {
            return matcher.start();
        } else {
            return -1;
        }
    }

    private boolean enabled = true;
    private Context context;
    private FormEnvironment formEnvironment;
    private boolean visible = true;

    @Override
    public void setContext(Context context) {
        this.context = context;
    }

    /**
     * Returns the {@code Context} that the framework has associated with
     * this form component in {@code setContext}.
     *
     * @return the installer context.
     */
    public Context getContext() {
        return context;
    }

    @Override
    public void setFormEnvironment(FormEnvironment formEnvironment) {
        this.formEnvironment = formEnvironment;
    }

    /**
     * Returns the {@code FormEnvironment} that the framework has associated with
     * this form component in {@code setFormEnvironment}.
     *
     * @return the installer context.
     */
    @Override
    public FormEnvironment getFormEnvironment() {
        return formEnvironment;
    }

    @Override
    public boolean isEnabled() {
        return enabled;
    }

    @Override
    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
        Object configurationObject = getConfigurationObject();
        if (configurationObject instanceof JComponent) {
            ((JComponent)configurationObject).setEnabled(enabled);
        }
        if (formEnvironment instanceof FormEnvironmentImpl) {
            FormPanel formPanel = ((FormEnvironmentImpl)formEnvironment).getFormPanel();
            ScreenEnvelope screenEnvelope = formPanel.getScreenEnvelope();
            if (screenEnvelope != null) {
                screenEnvelope.formComponentEnabledChanged(this, enabled);
            }
        }
    }

    @Override
    public void setVisible(boolean visible) {
        this.visible = visible;
        if (formEnvironment != null) {
            ComponentTuple componentTuple = formEnvironment.getComponentTuple(this);
            if (componentTuple != null) {
                setVisible(componentTuple.getLeftComponent(), visible);
                setVisible(componentTuple.getCenterComponent(), visible);
                setVisible(componentTuple.getRightComponent(), visible);
                JComponent centerComponent = componentTuple.getCenterComponent();
                if (centerComponent != null) {
                    Container parent = centerComponent.getParent();
                    if (parent != null) {
                        parent.revalidate();
                    }
                }
            }
        }
    }

    private void setVisible(JComponent component, boolean visible) {
        if (component != null) {
            component.setVisible(visible);
        }
    }

    @Override
    public boolean isVisible() {
        return visible;
    }

    @Override
    public void migrateIds(Map<String, String> oldIdToNewId) {

    }

    @Override
    public void requestFocus() {
        Object configurationObject = getConfigurationObject();
        if (configurationObject instanceof JComponent) {
            final JComponent component = (JComponent)configurationObject;
            if (component.isFocusable()) {
                SwingUtilities.invokeLater(component::requestFocus);
            }
        }
    }

    @Override
    public boolean hasUserInput() {
        return true;
    }

    @Override
    public JComponent createLeftComponent() {
        return null;
    }

    @Override
    public boolean isFillCenterVertical() {
        return false;
    }

    @Override
    public JComponent createRightComponent() {
        return null;
    }

    @Override
    public Object getConfigurationObject() {
        return null;
    }

    @Override
    public Class<?> getConfigurationObjectClass() {
        return null;
    }

    @Override
    public boolean checkCompleted() {
        return true;
    }

    @Override
    public void initialize() {
    }

    // catch overridden methods with this name during compilation

    /**
     * The name of this method has changed to "initialize"
     */
    @SuppressWarnings("SpellCheckingInspection")
    @Deprecated
    protected final void initalize() {
    }

    @Override
    public void formWillActivate() {
    }

    @Override
    public void formActivated() {
    }

    @Override
    public void formDeactivated() {
    }

    @Override
    public boolean handleConsole(Console console) throws UserCanceledException {
        return true;
    }

    @Override
    public boolean handleUnattended() {
        return true;
    }

    /**
     * Helper method to get an init value. If the variable is defined and of class valueClass, the
     * variable value is used. Otherwise, the default value is used.
     * <p>
     * This method offers special handling for boolean, integer and enum values where a string value
     * will be parsed as well. In the case of boolean values, "true" and "false" in any case are
     * accepted as valid values. In the case of enum values, the enum constant name is accepted as
     * a string value.
     * </p>
     *
     * @param defaultValue the default value. Can be null.
     * @param variableName the installer variable name the value will be assigned to later on
     * @param valueClass   the desired class of the variable content
     * @return the init value to be used
     */
    protected Object getInitValue(Object defaultValue, String variableName, Class<?> valueClass) {
        Context context = getContext();
        Object value = context == null ? null : context.getVariable(variableName);
        if (value != null) {
            if (valueClass.isAssignableFrom(value.getClass())) {
                return value;
            } else if (value instanceof String) {
                String stringValue = (String)value;
                if (valueClass == Boolean.class) {
                    if (stringValue.equalsIgnoreCase("true") || stringValue.equalsIgnoreCase("false")) {
                        return Boolean.parseBoolean(stringValue);
                    }
                } else if (valueClass == Integer.class) {
                    try {
                        return Integer.parseInt(stringValue);
                    } catch (NumberFormatException ignored) {
                    }
                } else if (valueClass.isEnum()) {
                    Object[] enumConstants = valueClass.getEnumConstants();
                    for (Object o : enumConstants) {
                        Enum enumConstant = (Enum)o;
                        if (enumConstant.name().equals(stringValue)) {
                            return enumConstant;
                        }
                    }
                }
            }
        }
        return defaultValue;
    }

    /**
     * Same as {@link AbstractBean#replaceVariables(String)}, only with the additional replacement of form variables.
     * @see FormEnvironment#replaceFormVariables(String, ReplacementMode, VariableErrorHandlingDescriptor)
     */
    public String replaceFormVariables(String value) throws UndefinedVariableException {
        return replaceFormVariables(value, ReplacementMode.PLAIN, VariableErrorHandlingDescriptor.DEFAULT);
    }

    /**
     * Same as {@link AbstractBean#replaceVariables(String, ReplacementMode)}, only with the additional replacement of form variables.
     * @see FormEnvironment#replaceFormVariables(String, ReplacementMode, VariableErrorHandlingDescriptor)
     */
    public String replaceFormVariables(String value, ReplacementMode replacementMode) throws UndefinedVariableException {
        return replaceFormVariables(value, replacementMode, VariableErrorHandlingDescriptor.DEFAULT);
    }

    /**
     * Same as {@link AbstractBean#replaceVariables(String, VariableErrorHandlingDescriptor)}, only with the additional replacement of form variables.
     * @see FormEnvironment#replaceFormVariables(String, ReplacementMode, VariableErrorHandlingDescriptor)
     */
    public String replaceFormVariables(String value, VariableErrorHandlingDescriptor errorHandlingDescriptor) throws UndefinedVariableException {
        return replaceFormVariables(value, ReplacementMode.PLAIN, errorHandlingDescriptor);
    }

    /**
     * Same as {@link AbstractBean#replaceVariables(String, ReplacementMode, VariableErrorHandlingDescriptor)}, only with the additional replacement of form variables.
     * @see FormEnvironment#replaceFormVariables(String, ReplacementMode, VariableErrorHandlingDescriptor)
     */
    public String replaceFormVariables(String value, ReplacementMode replacementMode, VariableErrorHandlingDescriptor errorHandlingDescriptor) throws UndefinedVariableException {
        if (formEnvironment != null) {
            return formEnvironment.replaceFormVariables(value, replacementMode, errorHandlingDescriptor);
        } else {
            return InstallerVariables.replaceVariables(value, replacementMode, errorHandlingDescriptor);
        }
    }

    /**
     * Same as {@link AbstractBean#replaceVariables(String[])}, only with the additional replacement of form variables.
     * @see FormEnvironment#replaceFormVariables(String, ReplacementMode, VariableErrorHandlingDescriptor)
     */
    public String[] replaceFormVariables(String[] values) throws UndefinedVariableException {
        return replaceFormVariables(values, VariableErrorHandlingDescriptor.DEFAULT);
    }

    /**
     * Same as {@link AbstractBean#replaceVariables(String[], VariableErrorHandlingDescriptor)}, only with the additional replacement of form variables.
     * @see FormEnvironment#replaceFormVariables(String, ReplacementMode, VariableErrorHandlingDescriptor)
     */
    public String[] replaceFormVariables(String[] values, VariableErrorHandlingDescriptor errorHandlingDescriptor) throws UndefinedVariableException {
        if (formEnvironment instanceof FormEnvironmentImpl) {
            Map<String, VariableReplacer> variableReplacers = new HashMap<>();
            variableReplacers.put(InstallerVariables.FORM_PREFIX, (FormEnvironmentImpl)formEnvironment);
            return InstallerVariables.replaceVariables(values, errorHandlingDescriptor, variableReplacers);
        } else {
            return InstallerVariables.replaceVariables(values, errorHandlingDescriptor);
        }
    }
}
