package com.install4j.api.events;

import com.install4j.api.context.Context;

/**
 * The event object that is passed to installer event listeners when the file installation is started.
 * Event listeners are registered in the {@link com.install4j.api.context.InstallerContext}.
 * <p>This event object contains additional information about the installed files.
 * The event type {@link EventType#FILE_INSTALLATION_STARTED} produces this event object when fired.
 * @author ej-technologies GmbH
 * @see InstallerEventListener
 * @see com.install4j.api.context.InstallerContext#addInstallerEventListener(InstallerEventListener)
 *
 */
public class InstallerFileInstallationEvent extends InstallerEvent {

    private long totalSize;
    private long fileCount;

    /**
     * The constructor is called by the framework.
     * @param source the event source
     * @param context the context
     * @param type the event type.
     * @param totalSize the total size of the installation in bytes
     * @param fileCount the total file count
     */
    public InstallerFileInstallationEvent(Object source, Context context, EventType type, long totalSize, long fileCount) {
        super(source, context, type, true);
        this.totalSize = totalSize;
        this.fileCount = fileCount;
    }

    /**
     * Returns the total size of the installation in bytes
     * @return the size in bytes
     */
    public long getTotalSize() {
        return totalSize;
    }

    /**
     * Returns the total file count
     * @return the file count
     */
    public long getFileCount() {
        return fileCount;
    }

    @Override
    public String toString() {
        return super.toString() + " [total size: " + totalSize + " bytes, file count: "+ fileCount + "]";
    }

    @Override
    public String getVerbose() {
        return super.getVerbose() + ": total size: " + totalSize + " bytes; file count: " + fileCount;
    }
}
