package com.install4j.api.events;

import com.install4j.api.context.Context;

import java.io.File;
import java.net.URL;
import java.util.LinkedHashSet;
import java.util.Set;

/**
 * The event object that is passed to installer event listeners for download events types.
 * Event listeners are registered in the {@link com.install4j.api.context.InstallerContext}.
 * <p>This event object contains additional information about the download.
 * The event types {@link EventType#BEFORE_DOWNLOAD} and {@link EventType#AFTER_DOWNLOAD}
 * produce this event object when fired.
 * @author ej-technologies GmbH
 * @see InstallerEventListener
 * @see com.install4j.api.context.InstallerContext#addInstallerEventListener(InstallerEventListener)
 *
 */
public class InstallerDownloadEvent extends InstallerEvent {

    private URL url;
    private File downloadFile;
    private long downloadSize;

    private Set<DownloadProgressListener> downloadProgressListeners = new LinkedHashSet<>();

    /**
     * The constructor is called by the framework.
     * @param source the event source
     * @param context the context
     * @param type the event type.
     * @param success whether the operation associated with the event was successful
     * @param url the download URL
     * @param downloadFile the download file
     * @param downloadSize the size of the download
     */
    public InstallerDownloadEvent(Object source, Context context, EventType type, boolean success, URL url, File downloadFile, long downloadSize) {
        super(source, context, type, success);
        init(url, downloadFile, downloadSize);
    }

    /**
     * The constructor is called by the framework.
     * @param source the event source
     * @param context the context
     * @param type the event type.
     * @param url the download URL
     * @param downloadFile the download file
     * @param downloadSize the size of the download
     */
    public InstallerDownloadEvent(Object source, Context context, EventType type, URL url, File downloadFile, long downloadSize) {
        super(source, context, type);
        init(url, downloadFile, downloadSize);
    }

    private void init(URL url, File destinationFile, long downloadSize) {
        this.url = url;
        this.downloadFile = destinationFile;
        this.downloadSize = downloadSize;
    }

    /**
     * Returns the URL of the download.
     * @return the URL
     */
    public URL getUrl() {
        return url;
    }

    /**
     * Returns the file where the download is saved.
     * @return the download file
     */
    public File getDownloadFile() {
        return downloadFile;
    }

    /**
     * Returns the size for the download in bytes.
     * @return the size
     */
    public long getDownloadSize() {
        return downloadSize;
    }

    /**
     * Add a progress listener that is informed how the progress download is progressing.
     * This only makes sense to add if the event type is {@link EventType#BEFORE_DOWNLOAD}.
     * @param listener the progress listener
     */
    public void addDownloadProgressListener(DownloadProgressListener listener) {
        downloadProgressListeners.add(listener);
    }

    /**
     * Get the complete set of download progress listeners.
     * @return the set
     */
    public Set<InstallerDownloadEvent.DownloadProgressListener> getDownloadProgressListeners() {
        return downloadProgressListeners;
    }

    @Override
    public String toString() {
        return super.toString() + " [url " + url + ", file " + downloadFile + ", size " + downloadSize +  "]";
    }

    @Override
    public String getVerbose() {
        String ret = super.getVerbose() + ": " + url + " to " + downloadFile;
        if (getType() == EventType.BEFORE_DOWNLOAD) {
            ret += "; size: " + downloadSize + " bytes";
        }
        return ret;
    }

    /**
     * A listener that is informed about the progress of a download.
     * This listener is registered on the event with
     * {@link #addDownloadProgressListener}.
     */
    public interface DownloadProgressListener {
        /**
         * Indicates download progress.
         * @param bytes how many bytes have been downloaded so far.
         */
        void downloadCompleted(long bytes);
    }
}
