package com.install4j.api.windows;

import com.install4j.api.context.UserCanceledException;
import com.install4j.runtime.installer.platform.win32.FileAssociations;

import java.io.File;

/**
 * Collection of static methods to create and remove file associations
 * on Microsoft Windows.
 * @author ej-technologies GmbH
 */
public class WinAssociations {
    /**
     * Creates a file association.
     * @param extension the extension including the leading dot (e.g. ".doc").
     * @param description a description of the file type.
     * @param iconFile the icon file for the document. Can be null.
     * @param executable the executable that shall be called with the file
     * as a parameter.
     */
    public static void create(String extension, String description, File iconFile, File executable) {
        try {
            FileAssociations.create(extension, description, iconFile, executable, null);
        } catch (UserCanceledException ignored) {
        }
    }

    /**
     * Checks whether a file association already exists.
     * @param extension the extension including the leading dot (e.g. ".doc").
     * @return true if an association exists
     */
    public static boolean exists(String extension) {
        return FileAssociations.exists(extension, RegistryRoot.HKEY_CURRENT_USER) || FileAssociations.exists(extension, RegistryRoot.HKEY_CLASSES_ROOT);
    }

    /**
     * Retrieves the executable associated with an extension.
     * @param extension the extension including the leading dot (e.g. ".doc").
     * @return the executable associated with the given extension. Null if there is no association.
     */
    public static String getExecutable(String extension) {
        String executable = FileAssociations.getExecutable(extension, RegistryRoot.HKEY_CURRENT_USER);
        if (executable != null) {
            return executable;
        } else {
            return FileAssociations.getExecutable(extension, RegistryRoot.HKEY_CLASSES_ROOT);
        }
    }


    /**
     * Removes a file association.
     * @param extension the extension including the leading dot (e.g. ".doc").
     */
    public static void remove(String extension) {
        try {
            FileAssociations.remove(extension, RegistryRoot.HKEY_CURRENT_USER, true);
            FileAssociations.remove(extension, RegistryRoot.HKEY_CLASSES_ROOT, true);
        } catch (UserCanceledException e) {
            e.printStackTrace();
        }
    }

    private WinAssociations() {
    }

}
