package com.install4j.api.context;

import org.jetbrains.annotations.Nls;

import java.io.File;

/**
 * The user notification interface during installation. It displays progress
 * and messages and can ask questions relevant for installation purposes
 * in custom actions.
 * <p>Status messages and progress are only displayed if the progress
 * interface is used by an action and the containing screen offers its own progress interface,
 *  like the "Installation" screen or the customizable "Display progress" screen.
 * @author ej-technologies GmbH
 * @see com.install4j.api.screens.Screen#getProgressInterface(ProgressInterface)
 * @see Context#getProgressInterface()
 */
public interface ProgressInterface {

    /**
     * Returned by {@code askOverwrite} if the user hits "No".
     */
    int OVERWRITE_NO = 1;

    /**
     * Returned by {@code askOverwrite} if the user hits "Yes".
     */
    int OVERWRITE_YES = 2;

    /**
     * Returned by {@code askOverwrite} if the user hits "All".
     */
    int OVERWRITE_ALL = 3;

    /**
     * Returned by {@code askOverwrite} if the user hits "Never".
     */
    int OVERWRITE_NEVER = 4;

    /**
     * Returned by {@code askRetry} if the user hits "No".
     */
    int RETRY_NO = 1;

    /**
     * Returned by {@code askRetry} if the user hits "Yes".
     */
    int RETRY_YES = 2;

    /**
     * Returned by {@code askRetry} if the user hits "Cancel".
     */
    int RETRY_CANCEL = 3;

    /**
     * Sets the status message on the progress screen.
     * <p>Note: Has no effect if not called from an action or if the containing screen doesn't display progress.
     * @param message the new message.
     */
    void setStatusMessage(@Nls String message);

    /**
     * Sets the detail message on the progress screen. This message is shown
     * below the status message.
     * <p>Note: Has no effect if not called from an action or if the containing screen doesn't display progress.
     * @param message the new message.
     */
    void setDetailMessage(@Nls String message);

    /**
     * Sets the completion percentage of the current action.
     * <p>Note: Has no effect if not called from an action or if the containing screen doesn't display progress.
     * @param value a value between 0 and 100.
     */
    void setPercentCompleted(int value);

    /**
     * Retrieves the current progress value
     * @return the current progress value between 0 and 100.
     *  Always returns 0 if not called from an action or if the containing screen doesn't display progress.
     */
    int getPercentCompleted();

    /**
     * Sets the completion percentage of a secondary task, like the unpacking of a JAR file that was packed with Pack200.
     * <p>Note: Has no effect if not called from an action or if the containing screen doesn't display progress.
     * @param value a value between 0 and 100.
     */
    void setSecondaryPercentCompleted(int value);

    /**
     * Indicate that an action is performed whose duration is not known.
     * <p>Note: Has no effect if not called from an action or if the containing screen doesn't display progress.
     * @param indeterminateProgress pass {@code true} to switch on indeterminate progress
     *        feedback and {@code false} to switch it off again. When you switch it off,
     *        the previously set percentage will be shown in the progress bar.
     */
    void setIndeterminateProgress(boolean indeterminateProgress);

    /**
     * Shows an error dialog with the specified message.
     * @param message the error message
     */
    void showFailure(String message);

    /**
     * Asks the user whether the specified file should be overwritten.
     * @param file the file in question.
     * @return one of the {@code OVERWRITE_*} constants.
     * @throws UserCanceledException if the user cancels the question
     */
    int askOverwrite(File file) throws UserCanceledException;

    /**
     * Asks the user whether the installer should retry to overwrite the
     * specified file.
     * @param file the file in question.
     * @return one of the {@code RETRY_*} constants
     * @throws UserCanceledException if the user cancels the question
     */
    int askRetry(File file) throws UserCanceledException;

    /**
     * Asks the user whether to continue after the specified file could not
     * be installed.
     * @param file the file in question.
     * @return {@code true} if the user wants to continue the installation.
     * @throws UserCanceledException if the user cancels the question
     */
    boolean askContinue(File file) throws UserCanceledException;

    /**
     * Returns if the installer application has been started in the "Unattended mode with progress dialog"
     * execution mode, either because the "Default execution mode" property of the installer application
     * was set accordingly or because {@code -splash &lt;title&gt;"} were passed as command line arguments.
     */
    default boolean isUnattendedProgressDialog() {
        return false;
    }

    /**
     * Returns if alerts are shown. This will return false if the installer application was started in the
     * "Unattended mode with progress dialog" execution mode as explained for {@link #isUnattendedProgressDialog()}
     * unless the "Show alerts" property was also set or {@code -alerts} was also passed as a command line argument.
     */
    default boolean isAlertsShown() {
        return true;
    }

}
