package com.install4j.api.beans;

import com.install4j.api.beaninfo.Install4JPropertyDescriptor;
import com.install4j.api.beaninfo.ScriptParameter;
import com.install4j.api.context.Context;
import com.install4j.api.context.InstallerContext;
import com.install4j.api.context.UserCanceledException;
import com.install4j.runtime.installer.ContextImpl;
import com.install4j.runtime.installer.InstallerVariables;

import java.io.File;

/**
 * Common base class for all abstract superclasses for beans that are handled by install4j.
 * It is not recommended to extend this class directly. Concrete superclasses exist for
 * <ul>
 * <li><a href="../screens/package-summary.html">screens</a></li>
 * <li><a href="../actions/package-summary.html">actions</a></li>
 * <li><a href="../formcomponents/package-summary.html">form components</a></li>
 * </ul>
 * <p>This class provides common utility methods for all bean types.
 * @author ej-technologies GmbH
 */
public abstract class AbstractBean implements Bean {

    /**
     * Same as {@link #replaceVariables(String, ReplacementMode, VariableErrorHandlingDescriptor)}, with
     * {@link ReplacementMode#PLAIN} as the replacement mode and
     * {@link VariableErrorHandlingDescriptor#DEFAULT} as the error handling descriptor.
     * @param value the original string
     * @return the string with all variables replaced.
     * @throws UndefinedVariableException if a variable name cannot be found and the error handling is set to
     * {@link VariableErrorHandling#EXCEPTION} for the variable type.
     */
    public static String replaceVariables(String value) throws UndefinedVariableException {
        return InstallerVariables.replaceVariables(value);
    }

    /**
     * Same as {@link #replaceVariables(String, ReplacementMode, VariableErrorHandlingDescriptor)}, with
     * {@link VariableErrorHandlingDescriptor#DEFAULT} as the error handling descriptor.
     * @param value the original string
     * @param replacementMode the replacement mode
     * @return the string with all variables replaced.
     * @throws UndefinedVariableException if a variable name cannot be found and the error handling is set to
     * {@link VariableErrorHandling#EXCEPTION} for the variable type.
     */
    public static String replaceVariables(String value, ReplacementMode replacementMode) throws UndefinedVariableException {
        return InstallerVariables.replaceVariables(value, replacementMode);
    }

    /**
     * Same as {@link #replaceVariables(String, ReplacementMode, VariableErrorHandlingDescriptor)}, with
     * {@link ReplacementMode#PLAIN} as the replacement mode.
     * @param value the original string
     * @param errorHandlingDescriptor describes how missing variables should be treated for each variable type
     * @return the string with all variables replaced.
     * @throws UndefinedVariableException if a variable name cannot be found and the error handling is set to
     * {@link VariableErrorHandling#EXCEPTION} for the variable type.
     */
    public static String replaceVariables(String value, VariableErrorHandlingDescriptor errorHandlingDescriptor) throws UndefinedVariableException  {
        return InstallerVariables.replaceVariables(value, ReplacementMode.PLAIN, errorHandlingDescriptor);
    }

    /**
     * Replace all installer variables and localization keys in a string.
     * <p>Note: compiler variables in the project are replaced at compile time. Compiler variables in external files,
     * such as a custom localization file, are not replaced at a compile time.
     * @param value the original string
     * @param replacementMode the replacement mode
     * @param errorHandlingDescriptor describes how missing variables should be treated for each variable type
     * @return the string with all variables replaced.
     * @throws UndefinedVariableException if a variable name cannot be found and the error handling is set to
     * {@link VariableErrorHandling#EXCEPTION} for the variable type.
     */
    public static String replaceVariables(String value, ReplacementMode replacementMode, VariableErrorHandlingDescriptor errorHandlingDescriptor) throws UndefinedVariableException {
        return InstallerVariables.replaceVariables(value, replacementMode, errorHandlingDescriptor);
    }

    /**
     * Replace all installer variables and localization keys in a file name.
     * The used error handling descriptor is {@link VariableErrorHandlingDescriptor#DEFAULT}.
     * <p>Note: compiler variables in the project are replaced at compile time. Compiler variables in external files,
     * such as a custom localization file, are not replaced at a compile time.
     * @param file the original file
     * @return the file with all variables replaced in its name.
     * @throws UndefinedVariableException if a variable name cannot be found and the error handling is set to
     * {@link VariableErrorHandling#EXCEPTION} for the variable type.
     */
    public static File replaceVariables(File file) throws UndefinedVariableException {
        return InstallerVariables.replaceVariables(file);
    }

    /**
     * Same as {@link #replaceVariables(String[], VariableErrorHandlingDescriptor)}, with
     * {@link VariableErrorHandlingDescriptor#DEFAULT} as the error handling descriptor.
     * @param values the original array
     * @return the array with all variables replaced
     * @throws UndefinedVariableException if a variable name cannot be found and the error handling is set to
     * {@link VariableErrorHandling#EXCEPTION} for the variable type.
     */
    public static String[] replaceVariables(String[] values) throws UndefinedVariableException {
        return replaceVariables(values, VariableErrorHandlingDescriptor.DEFAULT);
    }

    /**
     * Replace all installer variables and localization keys in a string array.
     * For array elements that consist of an installer variable that in turn contains
     * an array or a collection value, the elements of that array or collection value will be inserted into
     * the returned array. For this reason, the returned array may have more elements than the
     * original array. For example, if the array passed in as an argument has the elements:
     * <pre>{@code
     *   [0] = "One"
     *   [1] = "${installer:myVariable}"
     *   [2] = "Three"
     * }</pre>
     * and the variable {@code myVariable} has a String array value with the elements
     * <pre>{@code
     *   [0] = "Blue"
     *   [1] = "Green"
     * }</pre>
     * then the returned array will be
     * <pre>{@code
     *   [0] = "One"
     *   [1] = "Blue"
     *   [2] = "Green"
     *   [3] = "Three"
     * }</pre>
     *<p>
     * If the array in an installer variable is not of type {@code String[]} or the collection is not of type {@code List<String>},
     * each element will be converted to a string by calling {@code toString()} on it.
     * </p>
     * <p>Note: compiler variables are replaced at compile time.
     * @param values the original array
     * @param errorHandlingDescriptor describes how missing variables should be treated for each variable type
     * @return the array with all variables replaced
     * @throws UndefinedVariableException if a variable name cannot be found and the error handling is set to
     * {@link VariableErrorHandling#EXCEPTION} for the variable type.
     */
    public static String[] replaceVariables(String[] values, VariableErrorHandlingDescriptor errorHandlingDescriptor) throws UndefinedVariableException {
        return InstallerVariables.replaceVariables(values, errorHandlingDescriptor);
    }

    /**
     * Replace all installer variables and localization keys in a file array.
     * This method is analogous to {@link #replaceVariables(String[])}.
     * @param files the original array
     * @return the array with all variables replaced
     * @throws UndefinedVariableException if a variable name cannot be found and the error handling is set to
     * {@link VariableErrorHandling#EXCEPTION} for the variable type.
     */
    public static File[] replaceVariables(File[] files) throws UndefinedVariableException {
        if (files == null) {
            return null;
        } else {
            String[] replacedFileNames = InstallerVariables.replaceVariables(files, VariableErrorHandlingDescriptor.DEFAULT);
            File[] replacedFiles = new File[replacedFileNames.length];
            for (int i = 0; i < replacedFileNames.length; i++) {
                replacedFiles[i] = new File(replacedFileNames[i]);
            }
            return replacedFiles;
        }
    }

    /**
     * Synchronously execute an action list.
     * You should not use this method from a form component because it will block the EDT. Use
     * {@link #executeActionListAsync(ActionList, Object...)} instead.
     * @param actionList the action list object from your bean property.
     * @param extraScriptParameters if the action list provides extra script parameters in the BeanInfo by calling
     *          {@link com.install4j.api.beaninfo.ActionListPropertyDescriptor#setExtraScriptParameters(ScriptParameter[])},
     *                        you have to pass them here.
     * @return whether the list of actions was terminated due to an error. This can only happen if the user selected
     *         the "Break if an error occurs" check box in the configuration dialog of the action list, meaning that
     *         the {@link ActionList#isBreakOnError()} of the argument returns {@code true}.
     *         To detect errors in absence of this flag, call {@link Context#setErrorOccurred(boolean)}
     *         with an argument of {@code false} before calling this method and check the result of
     *         {@link Context#isErrorOccurred()} afterwards.
     */
    public static boolean executeActionListSync(ActionList actionList, Object... extraScriptParameters) throws UserCanceledException {
        return ContextImpl.getSingleContextInt().executeActionListSync(actionList, extraScriptParameters);
    }

    /**
     * Asynchronously execute an action list.
     * Do not invoke this method from an action, it will throw an {@code IllegalStateException} in that case. For
     * actions, use {@link #executeActionListSync(ActionList, Object...)} instead.
     * This method is intended to be used form a form component and will disable all components on the current screen
     * while the action list is being executed.
     * @param actionList the action list object from your bean property.
     * @param extraScriptParameters if the action list provides extra script parameters in the BeanInfo by calling
     *          {@link com.install4j.api.beaninfo.ActionListPropertyDescriptor#setExtraScriptParameters(ScriptParameter[])},
     *                        you have to pass them here.
     */
    public static void executeActionListAsync(ActionList actionList, Object... extraScriptParameters) {
        ContextImpl.getSingleContextInt().executeActionListAsync(actionList, extraScriptParameters);
    }

    /**
     * Roll back an action list.
     * This will only have an effect if called from the {@link com.install4j.api.actions.InstallAction#rollback(InstallerContext)}
     * method and if you called {@link #executeActionListSync(ActionList, Object...)} before
     * @param actionList the action list object from your bean property
     */
    @SuppressWarnings("unused")
    public static void rollbackActionList(ActionList actionList) {
        ContextImpl.getSingleContextInt().rollbackActionList(actionList);
    }

    /**
     * Get the value for a property for which a text replacement has been configured in the IDE.<p>
     * For properties with non-String values of certain types, you can choose to switch to text entry mode
     * in the context menu. This mechanism has to be enabled by calling {@link Install4JPropertyDescriptor#setAllowTextOverride(boolean)}
     * with {@code true} in the property descriptor.<p>
     * The user can set the text value to an installer variable expression or to a compiler
     * variable expression. In the getter of the property, the {@code replaceWithTextOverride()} method with the
     * appropriate return type has to be called to perform the actual replacement. All these instance methods call this
     * static method, so the feature also works for classes that do not extend {@code AbstractBean}.
     *
     * @param bean the bean for which the replacement should be performed
     * @param propertyName the property whose value should be replaced
     * @param resultType the type of the property
     * @return the replaced value or null if no replacement has been configured
     */
    public static <T> T getTextOverrideValue(Bean bean, String propertyName, Class<T> resultType) {
        return InstallerVariables.getTextOverrideValue(bean, propertyName, resultType);
    }

    /**
     * A convenience method for {@link #getTextOverrideValue} that returns a default value if no replacement has been
     * configured.
     */
    @SuppressWarnings("unchecked")
    // Do not add <T> to the Class parameter, it will produce unchecked warnings during compilation
    public static <T> T replaceWithTextOverride(Bean bean, String propertyName, T defaultValue, Class resultType) {
        T textOverride = getTextOverrideValue(bean, propertyName, (Class<T>)resultType);
        return textOverride != null ? textOverride : defaultValue;
    }

    /**
     * @see #replaceWithTextOverride(Bean, String, Object, Class)
     */
    @SuppressWarnings("unused")
    // Do not add <T> to the Class parameter, it will produce unchecked warnings during compilation
    protected <T> T replaceWithTextOverride(String propertyName, T defaultValue, Class resultType) {
        return replaceWithTextOverride(this, propertyName, defaultValue, resultType);
    }

    /**
     * @see #replaceWithTextOverride(Bean, String, Object, Class)
     */
    @SuppressWarnings("unused")
    protected boolean replaceWithTextOverride(String propertyName, boolean defaultValue) {
        Boolean overrideValue = getTextOverrideValue(this, propertyName, Boolean.class);
        return overrideValue != null ? overrideValue : defaultValue;
    }

    /**
     * @see #replaceWithTextOverride(Bean, String, Object, Class)
     */
    @SuppressWarnings("unused")
    protected int replaceWithTextOverride(String propertyName, int defaultValue) {
        Integer overrideValue = getTextOverrideValue(this, propertyName, Integer.class);
        return overrideValue != null ? overrideValue : defaultValue;
    }

    /**
     * @see #replaceWithTextOverride(Bean, String, Object, Class)
     */
    @SuppressWarnings("unused")
    protected long replaceWithTextOverride(String propertyName, long defaultValue) {
        Long overrideValue = getTextOverrideValue(this, propertyName, Long.class);
        return overrideValue != null ? overrideValue : defaultValue;
    }

    /**
     * @see #replaceWithTextOverride(Bean, String, Object, Class)
     */
    @SuppressWarnings("unused")
    protected short replaceWithTextOverride(String propertyName, short defaultValue) {
        Short overrideValue = getTextOverrideValue(this, propertyName, Short.class);
        return overrideValue != null ? overrideValue : defaultValue;
    }

    /**
     * @see #replaceWithTextOverride(Bean, String, Object, Class)
     */
    @SuppressWarnings("unused")
    protected byte replaceWithTextOverride(String propertyName, byte defaultValue) {
        Byte overrideValue = getTextOverrideValue(this, propertyName, Byte.class);
        return overrideValue != null ? overrideValue : defaultValue;
    }

    /**
     * @see #replaceWithTextOverride(Bean, String, Object, Class)
     */
    @SuppressWarnings("unused")
    protected char replaceWithTextOverride(String propertyName, char defaultValue) {
        Character overrideValue = getTextOverrideValue(this, propertyName, Character.class);
        return overrideValue != null ? overrideValue : defaultValue;
    }

    /**
     * @see #replaceWithTextOverride(Bean, String, Object, Class)
     */
    @SuppressWarnings("unused")
    protected float replaceWithTextOverride(String propertyName, float defaultValue) {
        Float overrideValue = getTextOverrideValue(this, propertyName, Float.class);
        return overrideValue != null ? overrideValue : defaultValue;
    }

    /**
     * @see #replaceWithTextOverride(Bean, String, Object, Class)
     */
    @SuppressWarnings("unused")
    protected double replaceWithTextOverride(String propertyName, double defaultValue) {
        Double overrideValue = getTextOverrideValue(this, propertyName, Double.class);
        return overrideValue != null ? overrideValue : defaultValue;
    }
}
